/*
  gtkmenu-standalone
  ==================
  (C) Will Brokenbourgh
  will_brokenbourgh@yahoo.com
  http://www.pismotek.com/brainout/
  - - - -
  Copyright (C) Will Brokenbourgh - will_brokenbourgh@yahoo.com

  Redistribution and use in source and binary forms, with or without modification,
  are permitted provided that the following conditions are met:

      Redistributions of source code must retain the above copyright notice,
      this list of conditions and the following disclaimer.

      Redistributions in binary form must reproduce the above copyright notice,
      this list of conditions and the following disclaimer in the documentation
      and/or other materials provided with the distribution.

      Neither the name of The author nor the names of its contributors may be used
      to endorse or promote products derived from this software without specific prior
      written permission.

  THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR
  IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
  AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR AND
  CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
  CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
  SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
  ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
  OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
  POSSIBILITY OF SUCH DAMAGE.
  - - - -
*/


#include <gtk/gtk.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <glib/gstdio.h>

/* version */
gchar strVersion[] = "0.2.0";

/* menulistitem declarations */
typedef struct MenuListItem MenuListItem;
struct MenuListItem {
    gchar strCategory[1024];
    gchar strExec[1024];
    gchar strIcon[1024];
    gchar strName[1024];
    gchar strNameLower[1024];

    gboolean blNeedsTerminal;
} mli[32766];

gint ml_idx = 0;

/* configuration variables */
typedef struct Prefs Prefs;
struct Prefs {
    gchar strTermRunCommand[1024];

    gchar strRunCommand[1024];
    gchar strExitCommand[1024];
    gchar strRebootCommand[1024];
    gchar strPoweroffCommand[1024];

    gint nLeft;
    gint nTop;
    gint nDockSize;
    gint nMenuHeight;
    gint nDelayTime;

    gboolean blShowGNOME;
    gboolean blShowLXDE;
    gboolean blShowKDE;
    gboolean blShowMATE;
    gboolean blShowXFCE;
} prefs;

/* misc variables */
gchar strP[2048] = {0};
GtkWidget *main_menu;

#ifdef __linux__
    gchar strSysApps[] = "/usr/share/applications";
    gchar strSysPix[] = "/usr/share/pixmaps";
#endif

#ifdef __FreeBSD__
    gchar strSysApps[] = "/usr/local/share/applications";
    gchar strSysPix[] = "/usr/local/share/pixmaps";
#endif

/* initialize preferences */
void init_prefs ()
{
    strcpy(prefs.strTermRunCommand, "x-terminal-emulator -e");
    strcpy(prefs.strRunCommand, "fbrun -nearmouse");
    strcpy(prefs.strExitCommand, "fluxbox-remote quit");
    strcpy(prefs.strRebootCommand, "sudo reboot");
    strcpy(prefs.strPoweroffCommand, "sudo poweroff");

    prefs.nLeft = -1;
    prefs.nTop = -1;
    prefs.nDockSize = 24;
    prefs.nMenuHeight = 0;
    prefs.nDelayTime = 0;

    prefs.blShowGNOME = TRUE;
    prefs.blShowLXDE = TRUE;
    prefs.blShowKDE = TRUE;
    prefs.blShowMATE = TRUE;
    prefs.blShowXFCE = TRUE;
}

/* string left-remove function */
void lremove(char * strP, int nStart)
{
    int nLen = strlen(strP);
    int i = 0;
    int nPos = 0;
    char strTemp[nLen + 1];

    memset(strTemp, 0, sizeof(strTemp));

    for (i = nStart; i < nLen; i++) {
        strTemp[nPos] = strP[i];
        nPos++;
    }

    strTemp[i + 1] = 0;


    strcpy(strP, strTemp);
}


/* string remove function */
void str_remove (gchar *strBase, gchar *strSearch)
{
    gchar *strTmp = NULL;

    strTmp = g_regex_replace_literal(g_regex_new(strSearch, 0, 0, NULL),
        strBase, -1, 1, "", G_REGEX_MATCH_NEWLINE_ANY, NULL);

    strcpy(strBase, strTmp);
}

/* clears/inits menulistitem */
void init_menu_list_item (MenuListItem mi)
{
    memset(mi.strCategory, 0, sizeof(mi.strCategory));
    memset(mi.strExec, 0, sizeof(mi.strExec));
    memset(mi.strIcon, 0, sizeof(mi.strIcon));
    memset(mi.strName, 0, sizeof(mi.strName));
    memset(mi.strNameLower, 0, sizeof(mi.strNameLower));
    mi.blNeedsTerminal = FALSE;
}

/* add menulistitem to mli array */
void add_to_list (MenuListItem mi)
{
    strcpy(mli[ml_idx].strCategory, mi.strCategory);
    strcpy(mli[ml_idx].strExec, mi.strExec);
    strcpy(mli[ml_idx].strIcon, mi.strIcon);
    strcpy(mli[ml_idx].strName, mi.strName);
    strcpy(mli[ml_idx].strNameLower, mi.strNameLower);
    mli[ml_idx].blNeedsTerminal = mi.blNeedsTerminal;

    ml_idx++;
}

/* convert string to all lower-case */
void to_lower (char * strIn)
{
    int nTemp = 0;
    int nLen = strlen(strIn);
    if (strIn == NULL) {
        return;
    }

    for (nTemp = 0; nTemp < nLen; nTemp++) {
      strIn[nTemp] = tolower(strIn[nTemp]);
    }
}

/* qsort compare function - sort by menuitem name */
int qsort_compare (const void * a, const void * b)
{
    MenuListItem *miTemp1 = (MenuListItem*)a;
    MenuListItem *miTemp2 = (MenuListItem*)b;

    return strcmp(miTemp1->strNameLower, miTemp2->strNameLower);
}

/* sort menuitemlist */
void sort_list ()
{
    qsort (mli, ml_idx, sizeof(MenuListItem), qsort_compare);
}

/* set menuitem's icon depending on strIcon contents */
void set_item_icon(GtkWidget *menuitem, gchar * strIcon)
{
    gboolean blHasPath = FALSE;
    gboolean blHasExt = FALSE;
    GtkWidget *img = NULL;
    GdkPixbuf *pbI = NULL;
    gchar strPixPath[512] = {0};
    GError *gerr = NULL;

    // check for absolute path or extension
    if (strchr(strIcon, '/') != NULL) {
        blHasPath = TRUE;
    }
    if (strstr(strIcon, ".png") != NULL ||
      strstr(strIcon, ".PNG") != NULL ||
      strstr(strIcon, ".svg") != NULL ||
      strstr(strIcon, ".SVG") != NULL ||
      strstr(strIcon, ".xpm") != NULL ||
      strstr(strIcon, ".XPM") != NULL
      ) {
        blHasExt = TRUE;
    }

    // no path or extension
    if (blHasPath == FALSE && blHasExt == FALSE) {
        gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(menuitem),
            GTK_WIDGET(gtk_image_new_from_icon_name(strIcon, GTK_ICON_SIZE_MENU)));
        return;
    }

    // has absolute path
    if (blHasPath == TRUE) {
        pbI = gdk_pixbuf_new_from_file_at_scale (strIcon, 16, 16, TRUE, &gerr);
        if (gerr != NULL) {
            img = gtk_image_new_from_icon_name("image-missing", GTK_ICON_SIZE_MENU);
        } else {
            img = gtk_image_new_from_pixbuf(pbI);
        }
        gtk_widget_set_size_request(img, 16, 16);
        gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(menuitem), GTK_WIDGET(img));
        return;
    }

    // has extension but NO absolute path
    if (blHasExt == TRUE && blHasPath == FALSE) {
        strcpy(strPixPath, strSysPix);
        strcat(strPixPath, "/");
        strcat(strPixPath, strIcon);

        pbI = gdk_pixbuf_new_from_file_at_scale (strPixPath, 16, 16, TRUE, &gerr);
        if (gerr != NULL) {
            memset(strPixPath, 0, sizeof(strPixPath));
            strcpy(strPixPath, strIcon);
            gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(menuitem),
                GTK_WIDGET(gtk_image_new_from_icon_name(strPixPath, GTK_ICON_SIZE_MENU)));
        } else {
            img = gtk_image_new_from_pixbuf(pbI);
        }
        gtk_widget_set_size_request(img, 16, 16);
        gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(menuitem), GTK_WIDGET(img));
        return;
    }
}

/* get menu height for positioning calculations */
void get_menu_size (GtkWidget *widget, GtkAllocation *allocation, void *data)
{
    prefs.nMenuHeight = allocation->height;
}

/* position menu properly */
void menu_positioner (GtkMenu *menu, gint *xx, gint *yy, gboolean *push_in, gpointer user_data)
{
    gint nScreenHeight = gdk_screen_get_height(gdk_screen_get_default());
    if (prefs.nLeft == -1) {
        prefs.nLeft = 0;
    }
    if (prefs.nTop == -1) {
        prefs.nTop = 0;
    }

    *push_in = TRUE;

    *xx = prefs.nLeft;

    if (prefs.nMenuHeight == 0) {
        *yy = 0;
    } else {
        *yy = ((nScreenHeight - prefs.nMenuHeight) - prefs.nDockSize) - prefs.nTop;
    }
}

/* display About box */
void show_about ()
{
    GtkWidget *diaAbout = NULL;
    diaAbout = gtk_about_dialog_new();
    gtk_about_dialog_set_program_name(GTK_ABOUT_DIALOG(diaAbout), "GtkMenu-Standalone");
    gtk_about_dialog_set_copyright(GTK_ABOUT_DIALOG(diaAbout), "(C) 2014 - Will Brokenbourgh");
    gtk_about_dialog_set_version(GTK_ABOUT_DIALOG(diaAbout), strVersion);
    gtk_about_dialog_set_comments(GTK_ABOUT_DIALOG(diaAbout), "Blog:\nwww.pismotek.com/brainout/");
    gtk_about_dialog_set_logo_icon_name(GTK_ABOUT_DIALOG(diaAbout), "gtk-about");
    gtk_dialog_run(GTK_DIALOG(diaAbout));
}

/* handle menuitem click */
void menuitem_click_handler (GtkMenuItem *menuitem, gpointer user_data)
{
    GError *gerr = NULL;

    if ((char*)user_data == NULL) {
        return;
    }

    g_chdir(getenv("HOME"));

    if (strcmp((char*)user_data, "...about...") == 0) {
        // show About dialog
        show_about();
    } else {
        // execute menuitem command
        g_spawn_command_line_async((char*)user_data, &gerr);
        if (gerr != NULL) {
            printf("gtkmenu-standalone - Error: %s\n", gerr->message);
        }
    }
    gtk_main_quit();
}

/* handle close of menu without selection */
void menu_closes (GtkMenuShell *menushell, gpointer user_data)
{
    gtk_main_quit();
}

/* parse config file or set defaults if none */
void process_config_file ()
{
    char strHome[1024] = {0};
    char cP = 0;
    int nPos = -1;

    strcpy(strHome, getenv("HOME"));
    strcat(strHome, "/.gtkmenu-standalone");

    FILE *fFile = NULL;

    fFile = fopen(strHome , "r");
    if (fFile == NULL) {
         printf("gtkmenu-standalone: Error opening config file - Using defaults\n");
         return;
    } else {
        while (feof(fFile) == FALSE) {

            nPos = -1;
            cP = 0;

            // read line
            while (cP != EOF) {
                cP = fgetc(fFile);

                nPos++;

                if (cP == '\n' ||
                    cP == '\r' ||
                    cP == 0 ||
                    cP == EOF
                    ) {
                    strP[nPos] = 0;
                    break;
                }

                strP[nPos] = cP;
            }

            // left positioning
            if (strncmp(strP, "left=", 5) == 0) {
                lremove(strP, 5);
                prefs.nLeft = atoi(strP);
            }
            // top positioning
            if (strncmp(strP, "top=", 4) == 0) {
                lremove(strP, 4);
                prefs.nTop = atoi(strP);
            }
            // users dock size
            if (strncmp(strP, "docksize=", 9) == 0) {
                lremove(strP, 9);
                prefs.nDockSize = atoi(strP);
            }
            // menu pop-up delay time
            if (strncmp(strP, "delay=", 6) == 0) {
                lremove(strP, 6);
                prefs.nDelayTime = atoi(strP);
            }
            // show GNOME items?
            if (strncmp(strP, "showgnomeitems=", 15) == 0) {
                lremove(strP, 15);
                to_lower(strP);

                if (strlen(strP) != 0) {
                    if (strstr(strP, "true") == NULL) {
                        prefs.blShowGNOME = TRUE;
                    } else {
                        prefs.blShowGNOME = FALSE;
                    }
                }
            }
            // show KDE items?
            if (strncmp(strP, "showkdeitems=", 13) == 0) {
                lremove(strP, 13);
                to_lower(strP);

                if (strlen(strP) != 0) {
                    if (strstr(strP, "true") == NULL) {
                        prefs.blShowKDE = TRUE;
                    } else {
                        prefs.blShowKDE = FALSE;
                    }
                }
            }
            // show LXDE items?
            if (strncmp(strP, "showlxdeitems=", 14) == 0) {
                lremove(strP, 14);
                to_lower(strP);

                if (strlen(strP) != 0) {
                    if (strstr(strP, "true") == NULL) {
                        prefs.blShowLXDE = TRUE;
                    } else {
                        prefs.blShowLXDE = FALSE;
                    }
                }
            }
            // show MATE items?
            if (strncmp(strP, "showmateitems=", 14) == 0) {
                lremove(strP, 14);
                to_lower(strP);

                if (strlen(strP) != 0) {
                    if (strstr(strP, "true") == NULL) {
                        prefs.blShowMATE = TRUE;
                    } else {
                        prefs.blShowMATE = FALSE;
                    }
                }
            }
            // show XFCE items?
            if (strncmp(strP, "showxfceitems=", 14) == 0) {
                lremove(strP, 14);
                to_lower(strP);

                if (strlen(strP) != 0) {
                    if (strstr(strP, "true") == NULL) {
                        prefs.blShowXFCE = TRUE;
                    } else {
                        prefs.blShowXFCE = FALSE;
                    }
                }
            }
            // custom menu items
            if (strncmp(strP, "custom=", 7) == 0) {
                lremove(strP, 7);

                gchar *strTok = NULL;
                int nIdx = 0;
                MenuListItem mTmp;

                if (strP != NULL) {
                    init_menu_list_item(mTmp);

                    strcpy(mTmp.strCategory, "CustomItems");

                    strTok = strtok(strP, "|");
                    while (strTok != NULL) {
                        switch (nIdx) {
                            case 0:
                                strcpy(mTmp.strName, strTok);
                                break;
                            case 1:
                                strcpy(mTmp.strExec, strTok);
                                break;
                            case 2:
                                strcpy(mTmp.strIcon, strTok);
                                add_to_list(mTmp);
                                break;
                            default:
                                break;
                        }
                        nIdx++;

                        strTok = strtok(NULL,"|");
                    }
                }
            }
            // run command
            if (strncmp(strP, "runcommand=", 11) == 0) {
                lremove(strP, 11);
                if (strlen(strP) != 0) {
                    strcpy(prefs.strRunCommand, strP);
                }
            }
            // exit command
            if (strncmp(strP, "exitcommand=", 12) == 0) {
                lremove(strP, 12);
                if (strlen(strP) != 0) {
                    strcpy(prefs.strExitCommand, strP);
                }
            }
            // reboot command
            if (strncmp(strP, "restartcommand=", 15) == 0) {
                lremove(strP, 15);
                if (strlen(strP) != 0) {
                    strcpy(prefs.strRebootCommand, strP);
                }
            }
            // poweroff command
            if (strncmp(strP, "poweroffcommand=", 16) == 0) {
                lremove(strP, 16);
                if (strlen(strP) != 0) {
                    strcpy(prefs.strPoweroffCommand, strP);
                }
            }
            // terminal run command
            if (strncmp(strP, "terminalruncommand=", 19) == 0) {
                lremove(strP, 19);
                if (strlen(strP) != 0) {
                    strcpy(prefs.strTermRunCommand, strP);
                }
            }
        }

        // close file
        if (fFile != NULL) {
            fclose(fFile);
        }
    }

}

/* parse .desktop files in /usr/share/applications and ~.local/share/applications */
void process_desktop_files ()
{
    gchar cP = 0;
    gchar strExecTemp[1024] = {0};
    int nPos = 0;
    gchar strHome[1024] = {0};
    MenuListItem mi;
    gboolean blFirstLoop = TRUE;
    gchar strPathTemp[1024] = {0};
    gchar strNameLowerTemp[1024] = {0};

    GDir *gdir = g_dir_open(strSysApps, 0, NULL);
    if (gdir == NULL) {
        printf("gtkmenu-standalone: Error opening /usr/share/applications - aborting");
        return;
    }

    FILE *fFile = NULL;

    // read desktop files, create and store menu items
    while (TRUE) {
        gboolean blNoShow = FALSE;
        gboolean blOkToParse = FALSE;

        init_menu_list_item(mi);

        memset(strPathTemp, 0, sizeof(strPathTemp));

        if (blFirstLoop) {
            strcpy(strPathTemp, strSysApps);
        } else {
            strcpy(strPathTemp, strHome);
            strcat(strPathTemp, "/");
        }

        // read next filename in folder
        const gchar *strFileTemp = g_dir_read_name(gdir);

        if (strFileTemp != NULL) {
            strcat(strPathTemp, "/");
            strcat(strPathTemp, strFileTemp);
            fFile = fopen(strPathTemp, "r");
        } else {
            fFile = NULL;
        }

        // switch from system applications folder to user's (or end if in user's)
        if (fFile == NULL) {
            // switch to local app folder and start over
            if (blFirstLoop) {
                g_dir_close(gdir);
                strcpy(strHome, getenv("HOME"));
                strcat(strHome, "/.local/share/applications");
                blFirstLoop = FALSE;

                gdir = g_dir_open(strHome, 0, NULL);
                if (gdir == NULL) {
                    printf("gtkmenu-standalone: Error opening %s - aborting", strHome);
                    return;
                }
            } else {
                break;
            }
        } else {
            while (feof(fFile) == FALSE) {

                nPos = -1;
                cP = 0;

                // read line
                while (cP != EOF) {
                    cP = fgetc(fFile);

                    nPos++;

                    if (cP == '\n' ||
                        cP == '\r' ||
                        cP == 0 ||
                        cP == EOF
                        ) {
                        strP[nPos] = 0;
                        break;
                    }

                    strP[nPos] = cP;
                }

                // set parse flag if we are in the Desktop Entry section
                if (strncmp(strP, "[", 1) == 0 && strstr(strP, "Desktop Entry") != NULL) {
                    blOkToParse = TRUE;
                }

                // parse line, if enabled and it isn't a comment
                if (blOkToParse == TRUE && strncmp(strP, "#", 1) != 0) {
                    // app name
                    if (strncmp(strP, "Name=", 5) == 0) {
                        lremove(strP, 5);
                        strcpy(mi.strName, strP);

                        // lower-case name (for sorting)
                        strcpy(strNameLowerTemp, strP);
                        to_lower(strNameLowerTemp);
                        strcpy(mi.strNameLower, strNameLowerTemp);
                    }
                    // app icon
                    if (strncmp(strP, "Icon=", 5) == 0) {
                        lremove(strP, 5);
                        strcpy(mi.strIcon, strP);
                    }
                    // app exec line
                    if (strncmp(strP, "Exec=", 5) == 0) {
                        lremove(strP, 5);
                        str_remove(strP, "%F");
                        str_remove(strP, "%f");
                        str_remove(strP, "%U");
                        str_remove(strP, "%u");
                        strcpy(mi.strExec, strP);
                    }
                    // app category
                    if (strncmp(strP, "Categories=", 11) == 0) {
                        if (blFirstLoop) {
                            lremove(strP, 11);
                            strcpy(mi.strCategory, strP);
                        } else {
                            strcpy(mi.strCategory, "Other");
                        }
                    }
                    // needs terminal?
                    if (strncmp(strP, "Terminal=", 9) == 0) {
                        lremove(strP, 9);
                        to_lower(strP);

                        if (strstr(strP, "true") != NULL) {
                            mi.blNeedsTerminal = TRUE;
                        } else {
                            mi.blNeedsTerminal = FALSE;
                        }
                    }
                    // don't display?
                    if (strncmp(strP, "NoDisplay=", 10) == 0) {
                        lremove(strP, 10);
                        to_lower(strP);

                        if (strstr(strP, "true") != NULL) {
                            blNoShow = TRUE;
                        }
                    }
                    // show only in KDE?
                    if (strstr(strP, "OnlyShowIn=KDE")) {
                        blNoShow = TRUE;
                    }
                    // stop parsing if there are other sections
                    if (strncmp(strP, "[", 1) == 0 && strstr(strP, "Desktop Entry") == NULL) {
                        break;
                    }
                }
            }

            // clean up and add to menuitemlist
            if (strlen(mi.strName) > 0  && !blNoShow) {
                if (mi.blNeedsTerminal) {
                    memset(strExecTemp, 0, sizeof(strExecTemp));
                    strcpy(strExecTemp, mi.strExec);
                    strcpy(mi.strExec, prefs.strTermRunCommand);
                    strcat(mi.strExec, " ");
                    strcat(mi.strExec, strExecTemp);
                }
                add_to_list(mi);
            }

        }

        // close file
        if (fFile != NULL) {
            fclose(fFile);
        }

    }
}

/* start building the gtkmenu from menulistitems */
void build_menu ()
{
    GtkWidget *menTemp;
    int nI = 0;
    GtkWidget *menu_item;
    char strCategoryTemp[512] = {0};

    // category used flags
    gboolean blCustomUsed = FALSE;
    gboolean blPrefUsed = FALSE;
    gboolean blAccUsed = FALSE;
    gboolean blDevelUsed = FALSE;
    gboolean blEduUsed = FALSE;
    gboolean blGameUsed = FALSE;
    gboolean blGraphUsed = FALSE;
    gboolean blInterUsed = FALSE;
    gboolean blMultiUsed = FALSE;
    gboolean blOfficUsed = FALSE;
    gboolean blSciUsed = FALSE;
    gboolean blSysUsed = FALSE;
    gboolean blHamUsed = FALSE;

    // create menu and set up signals
    main_menu = gtk_menu_new ();
    gtk_menu_set_screen (GTK_MENU(main_menu), gdk_screen_get_default ());

    // create category menus
    GtkWidget *menPref = gtk_menu_new();
    GtkWidget *menAcc = gtk_menu_new();
    GtkWidget *menDevel = gtk_menu_new();
    GtkWidget *menEdu = gtk_menu_new();
    GtkWidget *menGame = gtk_menu_new();
    GtkWidget *menGraph = gtk_menu_new();
    GtkWidget *menInter = gtk_menu_new();
    GtkWidget *menMulti = gtk_menu_new();
    GtkWidget *menOffic = gtk_menu_new();
    GtkWidget *menOth = gtk_menu_new();
    GtkWidget *menSci = gtk_menu_new();
    GtkWidget *menSys = gtk_menu_new();
    GtkWidget *menHam = gtk_menu_new();

    // *** collate by standard categories ***
    for (nI = 0; nI < ml_idx; nI++) {

        memset(strCategoryTemp, 0, sizeof(strCategoryTemp));
        strcpy(strCategoryTemp, mli[nI].strCategory);

        menTemp = menOth;

        // custom items
        if (strstr(strCategoryTemp, "CustomItems")) {
            menTemp = GTK_WIDGET(main_menu);
            blCustomUsed = TRUE;
        }

        // settings / preferences
        if (strstr(strCategoryTemp, "Settings")) {
            menTemp = menPref;
            blPrefUsed = TRUE;
        }

        // accessories / utility
        if (strstr(strCategoryTemp, "Accessories") ||
            strstr(strCategoryTemp, "Utility")
            ) {
            menTemp = menAcc;
            blAccUsed = TRUE;
        }

        // development / programming
        if (strstr(strCategoryTemp, "Development")) {
            menTemp = menDevel;
            blDevelUsed = TRUE;
        }

        // education / science
        if (strstr(strCategoryTemp, "Education")) {
            menTemp = menEdu;
            blEduUsed = TRUE;
        }

        // games
        if (strstr(strCategoryTemp, "Game")) {
            menTemp = menGame;
            blGameUsed = TRUE;
        }

        // graphics
        if (strstr(strCategoryTemp, "Graphics")) {
            menTemp = menGraph;
            blGraphUsed = TRUE;
        }

        // internet / network
        if (strstr(strCategoryTemp, "Internet") ||
            strstr(strCategoryTemp, "WebBrowser") ||
            strstr(strCategoryTemp, "Network")
            ) {
            menTemp = menInter;
            blInterUsed = TRUE;
        }

        // multimedia / sound/video
        if (strstr(strCategoryTemp, "Audio") ||
            strstr(strCategoryTemp, "AudioVideo") ||
            strstr(strCategoryTemp, "Multimedia") ||
            strstr(strCategoryTemp, "Video")
            ) {
            menTemp = menMulti;
            blMultiUsed = TRUE;
        }

        // office / productivity
        if (strstr(strCategoryTemp, "Office")  ||
            strstr(strCategoryTemp, "Productivity")  ||
            strstr(strCategoryTemp, "WordProcessor")
            ) {
            menTemp = menOffic;
            blOfficUsed = TRUE;
        }

        // science
        if (strstr(strCategoryTemp, "Science"))
        {
            menTemp = menSci;
            blSciUsed = TRUE;
        }

        // system
        if (strstr(strCategoryTemp, "System"))
        {
            menTemp = menSys;
            blSysUsed = TRUE;
        }

        // ham radio
        if (strstr(strCategoryTemp, "HamRadio"))
        {
            menTemp = menHam;
            blHamUsed = TRUE;
        }

       // add item to menu
        menu_item = gtk_image_menu_item_new_with_label(mli[nI].strName);
        set_item_icon(menu_item, mli[nI].strIcon);
        g_signal_connect(GTK_WIDGET(menu_item), "activate", G_CALLBACK(menuitem_click_handler), mli[nI].strExec);
        gtk_menu_shell_append(GTK_MENU_SHELL(menTemp), menu_item);
    }

    // custom items used
    if (blCustomUsed == TRUE) {
        gtk_menu_shell_append(GTK_MENU_SHELL(main_menu), gtk_separator_menu_item_new());
    }

    // prefs used
    if (blPrefUsed == TRUE) {
        GtkWidget *catPref = gtk_image_menu_item_new_with_label("Preferences");
        gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(catPref),
            GTK_WIDGET(gtk_image_new_from_icon_name("preferences-desktop", GTK_ICON_SIZE_MENU)));
        gtk_menu_item_set_submenu(GTK_MENU_ITEM(catPref), menPref);
        gtk_menu_shell_append(GTK_MENU_SHELL(main_menu), catPref);
        gtk_menu_shell_append(GTK_MENU_SHELL(main_menu), gtk_separator_menu_item_new());
    }

    // acc used
    if (blAccUsed == TRUE) {
        GtkWidget *catAcc = gtk_image_menu_item_new_with_label("Accessories");
        gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(catAcc),
            GTK_WIDGET(gtk_image_new_from_icon_name("applications-accessories", GTK_ICON_SIZE_MENU)));
        gtk_menu_item_set_submenu(GTK_MENU_ITEM(catAcc), menAcc);
        gtk_menu_shell_append(GTK_MENU_SHELL(main_menu), catAcc);
    }

    // devel used
    if (blDevelUsed == TRUE) {
        GtkWidget *catDevel = gtk_image_menu_item_new_with_label("Development");
        gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(catDevel),
            GTK_WIDGET(gtk_image_new_from_icon_name("applications-development", GTK_ICON_SIZE_MENU)));
        gtk_menu_item_set_submenu(GTK_MENU_ITEM(catDevel), menDevel);
        gtk_menu_shell_append(GTK_MENU_SHELL(main_menu), catDevel);
    }

    // edu used
    if (blEduUsed == TRUE) {
        GtkWidget *catEdu = gtk_image_menu_item_new_with_label("Education");
        gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(catEdu),
            GTK_WIDGET(gtk_image_new_from_icon_name("applications-science", GTK_ICON_SIZE_MENU)));
        gtk_menu_item_set_submenu(GTK_MENU_ITEM(catEdu), menEdu);
        gtk_menu_shell_append(GTK_MENU_SHELL(main_menu), catEdu);
    }

    // game used
    if (blGameUsed == TRUE) {
        GtkWidget *catGame = gtk_image_menu_item_new_with_label("Games");
        gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(catGame),
            GTK_WIDGET(gtk_image_new_from_icon_name("applications-games", GTK_ICON_SIZE_MENU)));
        gtk_menu_item_set_submenu(GTK_MENU_ITEM(catGame), menGame);
        gtk_menu_shell_append(GTK_MENU_SHELL(main_menu), catGame);
    }

    // graph used
    if (blGraphUsed == TRUE) {
        GtkWidget *catGraph = gtk_image_menu_item_new_with_label("Graphics");
        gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(catGraph),
            GTK_WIDGET(gtk_image_new_from_icon_name("applications-graphics", GTK_ICON_SIZE_MENU)));
        gtk_menu_item_set_submenu(GTK_MENU_ITEM(catGraph), menGraph);
        gtk_menu_shell_append(GTK_MENU_SHELL(main_menu), catGraph);
    }

    // inter used
    if (blInterUsed == TRUE) {
        GtkWidget *catInter = gtk_image_menu_item_new_with_label("Network");
        gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(catInter),
            GTK_WIDGET(gtk_image_new_from_icon_name("applications-internet", GTK_ICON_SIZE_MENU)));
        gtk_menu_item_set_submenu(GTK_MENU_ITEM(catInter), menInter);
        gtk_menu_shell_append(GTK_MENU_SHELL(main_menu), catInter);
    }

    // ham used
    if (blHamUsed) {
        GtkWidget *catHam = gtk_image_menu_item_new_with_label("Ham Radio");
        gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(catHam),
            GTK_WIDGET(gtk_image_new_from_icon_name("applications-other", GTK_ICON_SIZE_MENU)));
        gtk_menu_item_set_submenu(GTK_MENU_ITEM(catHam), menHam);
        gtk_menu_shell_append(GTK_MENU_SHELL(main_menu), catHam);
    }

    // multi used
    if (blMultiUsed == TRUE) {
        GtkWidget *catMulti = gtk_image_menu_item_new_with_label("Multimedia");
        gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(catMulti),
            GTK_WIDGET(gtk_image_new_from_icon_name("applications-multimedia", GTK_ICON_SIZE_MENU)));
        gtk_menu_item_set_submenu(GTK_MENU_ITEM(catMulti), menMulti);
        gtk_menu_shell_append(GTK_MENU_SHELL(main_menu), catMulti);
    }

    // office used
    if (blOfficUsed == TRUE) {
        GtkWidget *catOffic = gtk_image_menu_item_new_with_label("Office");
        gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(catOffic),
            GTK_WIDGET(gtk_image_new_from_icon_name("applications-office", GTK_ICON_SIZE_MENU)));
        gtk_menu_item_set_submenu(GTK_MENU_ITEM(catOffic), menOffic);
        gtk_menu_shell_append(GTK_MENU_SHELL(main_menu), catOffic);
    }

    // other used (always enabled)
    GtkWidget *catOth = gtk_image_menu_item_new_with_label("Other");
    gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(catOth),
        GTK_WIDGET(gtk_image_new_from_icon_name("applications-other", GTK_ICON_SIZE_MENU)));
    gtk_menu_item_set_submenu(GTK_MENU_ITEM(catOth), menOth);
    gtk_menu_shell_append(GTK_MENU_SHELL(main_menu), catOth);

    // science used
    if (blSciUsed == TRUE) {
        GtkWidget *catSci = gtk_image_menu_item_new_with_label("Science");
        gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(catSci),
            GTK_WIDGET(gtk_image_new_from_icon_name("applications-science", GTK_ICON_SIZE_MENU)));
        gtk_menu_item_set_submenu(GTK_MENU_ITEM(catSci), menSci);
        gtk_menu_shell_append(GTK_MENU_SHELL(main_menu), catSci);
    }

    // sys used
    if (blSysUsed == TRUE) {
        GtkWidget *catSys = gtk_image_menu_item_new_with_label("System");
        gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(catSys),
            GTK_WIDGET(gtk_image_new_from_icon_name("applications-system", GTK_ICON_SIZE_MENU)));
        gtk_menu_item_set_submenu(GTK_MENU_ITEM(catSys), menSys);
        gtk_menu_shell_append(GTK_MENU_SHELL(main_menu), catSys);
    }

    // append about item
    gtk_menu_shell_append(GTK_MENU_SHELL(main_menu), gtk_separator_menu_item_new());

    // about
    menu_item = gtk_image_menu_item_new_with_label("About...");
    set_item_icon(menu_item, "gtk-about");
    g_signal_connect(GTK_WIDGET(menu_item), "activate", G_CALLBACK(menuitem_click_handler), "...about...");
    gtk_menu_shell_append(GTK_MENU_SHELL(main_menu), menu_item);

    /* append command items */
    gtk_menu_shell_append(GTK_MENU_SHELL(main_menu), gtk_separator_menu_item_new());

    // run
    menu_item = gtk_menu_item_new_with_label("Run...");
    g_signal_connect(GTK_WIDGET(menu_item), "activate", G_CALLBACK(menuitem_click_handler), prefs.strRunCommand);
    gtk_menu_shell_append(GTK_MENU_SHELL(main_menu), menu_item);

    // exit
    gtk_menu_shell_append(GTK_MENU_SHELL(main_menu), gtk_separator_menu_item_new());

    menu_item = gtk_image_menu_item_new_with_label("Exit");
    set_item_icon(menu_item, "system-log-out");
    g_signal_connect(GTK_WIDGET(menu_item), "activate", G_CALLBACK(menuitem_click_handler), prefs.strExitCommand);
    gtk_menu_shell_append(GTK_MENU_SHELL(main_menu), menu_item);

    // reboot
    menu_item = gtk_image_menu_item_new_with_label("Reboot");
    set_item_icon(menu_item, "view-refresh");
    g_signal_connect(GTK_WIDGET(menu_item), "activate", G_CALLBACK(menuitem_click_handler), prefs.strRebootCommand);
    gtk_menu_shell_append(GTK_MENU_SHELL(main_menu), menu_item);

    // shutdown
    menu_item = gtk_image_menu_item_new_with_label("Shutdown");
    set_item_icon(menu_item, "system-shutdown");
    g_signal_connect(GTK_WIDGET(menu_item), "activate", G_CALLBACK(menuitem_click_handler), prefs.strPoweroffCommand);
    gtk_menu_shell_append(GTK_MENU_SHELL(main_menu), menu_item);
}

/* display the menu */
gboolean popup_menu ()
{
    gtk_widget_show_all(GTK_WIDGET(main_menu));

    g_signal_connect(GTK_WIDGET(main_menu), "size-allocate", G_CALLBACK(get_menu_size), NULL);
    g_signal_connect(GTK_MENU_SHELL(main_menu), "deactivate", G_CALLBACK(menu_closes), NULL);
    g_signal_connect(GTK_MENU_SHELL(main_menu), "cancel", G_CALLBACK(menu_closes), NULL);

    gtk_menu_popup(GTK_MENU(main_menu), NULL, NULL, menu_positioner, NULL, 0, gtk_get_current_event_time());
    gtk_menu_reposition(GTK_MENU(main_menu));

    return FALSE;
}

/* kill app after a while */
gboolean times_up ()
{
    exit(0);
    return FALSE;
}

/* delayed start */
gboolean lets_go ()
{
    process_desktop_files();

    sort_list();

    build_menu();
    popup_menu();

    return FALSE;
}

/* Main program */
int main (int argc, char *argv[])
{
    gtk_init(&argc, &argv);

    // app timeout
    g_timeout_add(60000, times_up, NULL);

    init_prefs();
    process_config_file();

    if (prefs.nDelayTime < 0) {
        prefs.nDelayTime = 0;
    }
    // delay opening of menu
    g_timeout_add(prefs.nDelayTime, lets_go, NULL);

    gtk_main();

    return 0;
}
