/*
 * gtkmenu-standalone
 * ==================
 * (C) Will Brokenbourgh
 * will_brokenbourgh@yahoo.com
 * http://www.pismotek.com/brainout/
 * - - - -
 * Thank you Jesus!  To God be the glory!  C programming is fun!! :-D
 * - - - -
 * Copyright (C) 2014 - 2016 Will Brokenbourgh - will_brokenbourgh@yahoo.com
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */


/* includes */
#include <gtk/gtk.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <glib/gstdio.h>

/* app definitions */
#define APP_VERSION      "0.4.4"
#define APP_NAME         "Gtk Menu - Standalone"
#define APP_COPYRIGHT    "(C) 2014 - 2016 Will Brokenbourgh"
#define APP_URL          "http://www.pismotek.com/brainout/"
#define APP_ICON         "gtk-about"

/* size definitions */
#define MAX_PROP_LEN          1024
#define MAX_READ_LINE_LEN     5000
#define MAX_LIST_SIZE        32766
#define MAX_CUSTOM_LIST_SIZE   100


/* menulistitem declarations */
typedef struct tagMenuListItem MenuListItem;
struct tagMenuListItem {
    gchar strCategory[MAX_PROP_LEN];
    gchar strExec[MAX_PROP_LEN];
    gchar strIcon[MAX_PROP_LEN];
    gchar strName[MAX_PROP_LEN];

    gboolean blNeedsTerminal;
};

/* application globals */
struct structApp {

    gchar strUserName[MAX_PROP_LEN];
    
    gchar strAppsPaths[MAX_CUSTOM_LIST_SIZE][MAX_PROP_LEN];
    gint apps_paths_idx;

    gchar strSysPixPath[MAX_CUSTOM_LIST_SIZE][MAX_PROP_LEN];
    gint pix_paths_idx;

    MenuListItem menuListItems[MAX_LIST_SIZE];

    gint menuListItems_idx;
    gchar strTermRunCommand[MAX_PROP_LEN];

    gchar strRunCommand[MAX_PROP_LEN];
    gchar strExitCommand[MAX_PROP_LEN];
    gchar strRebootCommand[MAX_PROP_LEN];
    gchar strPoweroffCommand[MAX_PROP_LEN];

    gint nLeft;
    gint nTop;
    gint nDockSize;
    gint nMenuHeight;
    gint nDelayTime;

    gboolean blShowGNOME;
    gboolean blShowLXDE;
    gboolean blShowKDE;
    gboolean blShowMATE;
    gboolean blShowXFCE;
    gboolean blShowCinnamon;
    gboolean blShowUnity;
    
    gboolean blRecurse;

    MenuListItem menuCustomItems[MAX_CUSTOM_LIST_SIZE];
    gint nMenuCustomItem_idx;

    GtkWidget* main_menu;

} app;


/* ###### functions ############### */

/* add slash to end of string, IF missing */
void assure_trailing_slash (gchar* strIn)
{
    if (strIn[strlen(strIn) - 1] != '/')
        strncat(strIn, "/", MAX_PROP_LEN);
}


/* check and add valid user, sys and pixmap paths */
void check_and_add_user_and_system_paths ()
{
    gchar strTmp1[MAX_PROP_LEN] = {0};
    gchar strTmp2[MAX_PROP_LEN] = {0};
    gint i = 0;

    for (i = 0; i < MAX_CUSTOM_LIST_SIZE; i++)
        strcpy(app.strAppsPaths[i], "");

    // set user name, if available (should be!)
    memset(app.strUserName, 0, sizeof(MAX_PROP_LEN));
    strncpy(app.strUserName, getenv("USER"), MAX_PROP_LEN);    

    // check and add system application dirs
    if (g_dir_open("/usr/share/applications", 0, NULL)) {
        strncpy(app.strAppsPaths[app.apps_paths_idx], "/usr/share/applications", MAX_PROP_LEN);
        app.apps_paths_idx++;
    }

    if (g_dir_open("/usr/local/share/applications", 0, NULL)) {
        strncpy(app.strAppsPaths[app.apps_paths_idx], "/usr/local/share/applications", MAX_PROP_LEN);
        app.apps_paths_idx++;
    }

    // add user's 'HOME/.local/share/applications' path, if it exists
    if (app.strUserName != NULL) {
        // typical Linux, OpenBSD home location
        sprintf(strTmp1, "/home/%s/.local/share/applications", app.strUserName);

        if (g_dir_open(strTmp1, 0, NULL)) {
            strncpy(app.strAppsPaths[app.apps_paths_idx], strTmp1, MAX_PROP_LEN);
            app.apps_paths_idx++;
        }

        // typical FreeBSD and others home location
        sprintf(strTmp2, "/usr/home/%s/.local/share/applications", app.strUserName);

        if (g_dir_open(strTmp2, 0, NULL)) {
            strncpy(app.strAppsPaths[app.apps_paths_idx], strTmp2, MAX_PROP_LEN);
            app.apps_paths_idx++;
        }
    }

    // check and add system pixmap dirs
    if (g_dir_open("/usr/share/pixmaps", 0, NULL)) {
        strncpy(app.strSysPixPath[app.pix_paths_idx], "/usr/share/pixmaps/", MAX_PROP_LEN);
        app.pix_paths_idx++;
    }

    if (g_dir_open("/usr/local/share/pixmaps", 0, NULL)) {
        strncpy(app.strSysPixPath[app.pix_paths_idx], "/usr/local/share/pixmaps/", MAX_PROP_LEN);
        app.pix_paths_idx++;
    }
}


/* initialize app variables */
void init_app ()
{
    app.apps_paths_idx = 0;
    app.pix_paths_idx = 0;
    
    check_and_add_user_and_system_paths();

    // set defaults for custom commands
    strncpy(app.strTermRunCommand, "x-terminal-emulator -e", MAX_PROP_LEN);
    strncpy(app.strRunCommand, "fbrun -nearmouse", MAX_PROP_LEN);
    strncpy(app.strExitCommand, "fluxbox-remote quit", MAX_PROP_LEN);
    strncpy(app.strRebootCommand, "sudo reboot", MAX_PROP_LEN);
    strncpy(app.strPoweroffCommand, "sudo poweroff", MAX_PROP_LEN);

    // desktop environment item prefs
    app.blShowGNOME = TRUE;
    app.blShowLXDE = TRUE;
    app.blShowKDE = TRUE;
    app.blShowMATE = TRUE;
    app.blShowXFCE = TRUE;
    app.blShowCinnamon = TRUE;
    app.blShowUnity = TRUE;

    // other options
    app.blRecurse = TRUE;
    app.nLeft = -1;
    app.nTop = -1;
    app.nDockSize = 24;
    app.nMenuHeight = 0;
    app.nDelayTime = 100;

    app.nMenuCustomItem_idx = 0;
}


/* return property and value from input */
void get_prop_and_val (gchar* strIn, gchar* strProp, gchar* strVal)
{
    gchar strProperty[MAX_PROP_LEN] = {0};
    gchar strValue[MAX_PROP_LEN] = {0};

    gint nIdx = 0;
    gint i = 0;

    // property
    for (i = 0; i < MAX_PROP_LEN; i++) {
        if (strIn[i] != '=' && strIn[i] != '\0') {
            strProperty[nIdx++] = strIn[i];
        } else {
            strProperty[nIdx++] = '\0';
            strncpy(strProp, strProperty, MAX_PROP_LEN);
            break;
        }
    }

    // value
    nIdx = 0;
    i++;

    for (; i < MAX_PROP_LEN; i++) {
        if (strIn[i] != 0) {
            strValue[nIdx++] = strIn[i];
        } else {
            strValue[nIdx++] = '\0';
            strncpy(strVal, strValue, MAX_PROP_LEN);
            return;
        }
    }
}


/* remove menu item exec parameters %U, etc */
void chop_params (gchar* strBase)
{
    if (strBase == NULL)
        return;

    gint nLen = strlen(strBase);
    gint nPos;    

    if (nLen < 1)
        return;

    // find the first '%', if any
    nPos = strcspn(strBase, "\%");

    if (nPos == nLen)
        return;

    // set the first '%' to a null character
    strBase[nPos] = '\0';
}


/* clears/inits menulistitem */
void init_menu_list_item (MenuListItem menuListItem)
{
    menuListItem.strCategory[0] = '\n';
    menuListItem.strExec[0] = '\n';
    menuListItem.strIcon[0] = '\n';
    menuListItem.strName[0] = '\n';
    menuListItem.blNeedsTerminal = FALSE;
}


/* add menulistitem to menuListItems array */
void add_to_list (MenuListItem menuListItem)
{
    strncpy(app.menuListItems[app.menuListItems_idx].strCategory, menuListItem.strCategory,
            MAX_PROP_LEN - 1);
    strncpy(app.menuListItems[app.menuListItems_idx].strExec, menuListItem.strExec, MAX_PROP_LEN - 1);
    strncpy(app.menuListItems[app.menuListItems_idx].strIcon, menuListItem.strIcon, MAX_PROP_LEN - 1);
    strncpy(app.menuListItems[app.menuListItems_idx].strName, menuListItem.strName, MAX_PROP_LEN - 1);
    app.menuListItems[app.menuListItems_idx].blNeedsTerminal = menuListItem.blNeedsTerminal;

    app.menuListItems_idx++;
}


/* qsort compare function - sort by menuitem name */
int qsort_compare (const void* a, const void* b)
{
    MenuListItem* miTemp1 = (MenuListItem*)a;
    MenuListItem* miTemp2 = (MenuListItem*)b;

    return strcmp(miTemp1->strName, miTemp2->strName);
}


/* sort menuitemlist */
void sort_list ()
{
    qsort(app.menuListItems, app.menuListItems_idx, sizeof(MenuListItem), qsort_compare);
}


/* set menuitem's icon depending on strIcon contents */
void set_item_icon(GtkWidget* menuitem, const gchar* strIcon)
{
    gboolean blHasPath = FALSE;
    gboolean blHasExt = FALSE;
    GtkWidget* img = NULL;
    GdkPixbuf* pbI = NULL;
    gchar strPixPath[MAX_PROP_LEN] = {0};
    GError* gerr = NULL;
    gint i = 0;
    gchar* strTemp = NULL;

    if (menuitem == NULL || strIcon == NULL)
        return;

    // normalize temp string to lower-case
    strTemp = g_ascii_strdown(strIcon, -1);

    // check for absolute path
    if (strchr(strTemp, '/') != NULL)
        blHasPath = TRUE;
    
    // check for extension (.png, .svg, etc)
    if (strstr(strTemp, ".png") != NULL ||
        strstr(strIcon, ".svg") != NULL ||
        strstr(strIcon, ".xpm") != NULL
      ) {
        blHasExt = TRUE;
    }

    // no path or extension
    if (blHasPath == FALSE && blHasExt == FALSE) {
        gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(menuitem),
                                      GTK_WIDGET(gtk_image_new_from_icon_name(strIcon,
                                      GTK_ICON_SIZE_MENU)));
        return;
    }

    // has absolute path
    if (blHasPath == TRUE) {
        
        pbI = gdk_pixbuf_new_from_file_at_scale (strIcon, 16, 16, TRUE, &gerr);

        if (gerr != NULL) {
            img = gtk_image_new_from_icon_name("image-missing", GTK_ICON_SIZE_MENU);
        } else {
            img = gtk_image_new_from_pixbuf(pbI);
        }
        
        if (GTK_IS_WIDGET(img)) {
            gtk_widget_set_size_request(GTK_WIDGET(img), 16, 16);
            gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(menuitem), GTK_WIDGET(img));
        }

        return;
    }

    // has extension but NO absolute path
    if (blHasExt == TRUE && blHasPath == FALSE) {

        for (i = 0; i < app.pix_paths_idx; i++) {            
            strncpy(strPixPath, app.strSysPixPath[i], MAX_PROP_LEN);
            strncat(strPixPath, "/", MAX_PROP_LEN);
            strncat(strPixPath, strIcon, MAX_PROP_LEN);

            pbI = gdk_pixbuf_new_from_file_at_scale(strPixPath, 16, 16, TRUE, &gerr);

            if (gerr != NULL) {
                memset(strPixPath, 0, sizeof(strPixPath));
                strncpy(strPixPath, strIcon, MAX_PROP_LEN);
                gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(menuitem),
                                              GTK_WIDGET(gtk_image_new_from_icon_name(strPixPath,
                                              GTK_ICON_SIZE_MENU)));
            } else {
                img = gtk_image_new_from_pixbuf(pbI);
            }
            
            if (GTK_IS_WIDGET(img)) {
                gtk_widget_set_size_request(GTK_WIDGET(img), 16, 16);
                gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(menuitem), GTK_WIDGET(img));
            }
        }
        return;
    }
}


/* get menu height for positioning calculations */
void get_menu_size (GtkWidget* widget, GtkAllocation* allocation)
{
    app.nMenuHeight = allocation->height;
}


/* position menu properly */
void menu_positioner (GtkMenu* menu, gint* xx, gint* yy, gboolean* push_in, gpointer user_data)
{
    gint nScreenHeight = gdk_screen_get_height(gdk_screen_get_default());

    if (app.nLeft == -1)
        app.nLeft = 0;

    if (app.nTop == -1)
        app.nTop = 0;

    *push_in = TRUE;

    *xx = app.nLeft;

    if (app.nMenuHeight == 0)
        *yy = 0;
    else
        *yy = ((nScreenHeight - app.nMenuHeight) - app.nDockSize) - app.nTop;
}


/* display About box */
void show_about_dialog ()
{
    GtkWidget* diaAbout = NULL;
    diaAbout = gtk_about_dialog_new();
    gtk_about_dialog_set_program_name(GTK_ABOUT_DIALOG(diaAbout), APP_NAME);
    gtk_about_dialog_set_copyright(GTK_ABOUT_DIALOG(diaAbout), APP_COPYRIGHT);
    gtk_about_dialog_set_version(GTK_ABOUT_DIALOG(diaAbout), APP_VERSION);
    gtk_about_dialog_set_comments(GTK_ABOUT_DIALOG(diaAbout), APP_URL);
    gtk_about_dialog_set_logo_icon_name(GTK_ABOUT_DIALOG(diaAbout), APP_ICON);
    
    gtk_dialog_run(GTK_DIALOG(diaAbout));
}


/* display Error dialog */
void show_error_dialog (gchar* strErrIn)
{   
    GtkWidget* diaErr = NULL;

    if (strErrIn == NULL)
        return;

    diaErr = gtk_message_dialog_new(NULL, GTK_DIALOG_MODAL, GTK_MESSAGE_ERROR, GTK_BUTTONS_OK, strErrIn);
    gtk_window_set_title(GTK_WINDOW(diaErr), "Gtk Menu - Error");

    gtk_dialog_run(GTK_DIALOG(diaErr));
}


/* handle menuitem click */
void menuitem_click_handler (GtkMenuItem* menuitem, gpointer user_data)
{
    GError* gerr = NULL;

    if (user_data == NULL)
        return;

    g_chdir(getenv("HOME"));

    if (strcmp((gchar*)user_data, "...about...") == 0) {
        // show About dialog
        show_about_dialog();
    } else {
        // execute menuitem command
        g_spawn_command_line_async((gchar*)user_data, &gerr);

        if (gerr != NULL)
            show_error_dialog(gerr->message);
    }

    gtk_main_quit();
}


/* parse config file or set defaults if none */
void process_config_file ()
{
    gchar strHome[MAX_PROP_LEN] = {0};
    gchar cP = 0;
    gint nPos = -1;
    gchar strP[MAX_READ_LINE_LEN] = {0};
    gchar strProp[MAX_PROP_LEN] = {0};    
    gchar strVal[MAX_PROP_LEN] = {0};

    FILE* fFile = NULL;
    
    // get user's config directory (usually ~/.config)
    strncpy(strHome, g_get_user_config_dir(), MAX_PROP_LEN);
    assure_trailing_slash(strHome);
    strncat(strHome, "gtkmenu-standalone/config", MAX_PROP_LEN);

    // attempt to open ~/.config/gtkmenu-standalone/config file
    fFile = fopen(strHome , "r");
    
    if (fFile == NULL) {
        // couldn't open config file, just use defaults
        printf("gtkmenu-standalone: Couldn't open config file.  Will use defaults.\n");
        return;
    } else {
        // opened config file, let's process it
        while (feof(fFile) == FALSE) {

            nPos = -1;
            cP = 0;

            // read characters from file, stop at eol or eof
            // so one (file) line is created
            while (cP != EOF) {                
                cP = fgetc(fFile);

                nPos++;

                if (cP == '\n' ||
                    cP == '\r' ||
                    cP == 0 ||
                    cP == EOF ||
                    nPos > (MAX_READ_LINE_LEN - 2)
                    ) {
                        strP[nPos] = '\0';
                        break;
                }

                strP[nPos] = cP;
            }
            
            // parse line and set 'property' and 'value'
            get_prop_and_val(strP, strProp, strVal);

            // normalize property name to lower-case
            strncpy(strProp, g_ascii_strdown(strProp, -1), MAX_PROP_LEN);

            // left positioning
            if (strcmp(strProp, "left") == 0) {
                app.nLeft = atoi(strVal);
            } else
            // top positioning
            if (strcmp(strProp, "top") == 0) {
                app.nTop = atoi(strVal);
            } else
            // users dock size
            if (strcmp(strProp, "docksize") == 0) {
                app.nDockSize = atoi(strVal);
            } else
            // menu pop-up delay time
            if (strcmp(strProp, "delay") == 0) {
                app.nDelayTime = atoi(strVal);
            } else
            // show GNOME items?
            if (strcmp(strProp, "gnome") == 0
                && strcmp(strVal, "false") == 0
                ) {
                    app.blShowGNOME = FALSE;
            } else
            // show KDE items?
            if (strcmp(strProp, "kde") == 0
                && strcmp(strVal, "false") == 0
                ) {
                    app.blShowKDE = FALSE;
            } else
            // show LXDE items?
            if (strcmp(strProp, "lxde") == 0
                && strcmp(strVal, "false") == 0
                ) {
                    app.blShowLXDE = FALSE;
            } else
            // show MATE items?
            if (strcmp(strProp, "mate") == 0
                && strcmp(strVal, "false") == 0
                ) {
                    app.blShowMATE = FALSE;
            } else
            // show XFCE items?
            if (strcmp(strProp, "xfce") == 0
                && strcmp(strVal, "false") == 0
                ) {
                    app.blShowXFCE = FALSE;
            } else
            // show Cinnamon items?
            if (strcmp(strProp, "cinnamon") == 0
                && strcmp(strVal, "false") == 0
                ) {
                    app.blShowCinnamon = FALSE;
            } else
            // show Unity items?
            if (strcmp(strProp, "unity") == 0
                && strcmp(strVal, "false") == 0
                ) {
                    app.blShowUnity = FALSE;
            } else
            // custom menu items
            if (strcmp(strProp, "custom") == 0) {

                gchar* strTok = NULL;
                int nIdx = 0;
                MenuListItem mTmp;

                if (strVal != NULL) {
                    init_menu_list_item(mTmp);

                    strncpy(mTmp.strCategory, "CustomItems", MAX_PROP_LEN);

                    strTok = strtok(strVal, "|");
                    while (strTok != NULL) {
                        switch (nIdx) {
                            case 0:
                                strncpy(mTmp.strName, strTok, MAX_PROP_LEN);
                                break;
                            case 1:
                                strncpy(mTmp.strExec, strTok, MAX_PROP_LEN);
                                break;
                            case 2:
                                strncpy(mTmp.strIcon, strTok, MAX_PROP_LEN);

                                strncpy(app.menuCustomItems[app.nMenuCustomItem_idx].strCategory,
                                        mTmp.strCategory, MAX_PROP_LEN);
                                strncpy(app.menuCustomItems[app.nMenuCustomItem_idx].strExec,
                                        mTmp.strExec, MAX_PROP_LEN);
                                strncpy(app.menuCustomItems[app.nMenuCustomItem_idx].strIcon,
                                        mTmp.strIcon, MAX_PROP_LEN);
                                strncpy(app.menuCustomItems[app.nMenuCustomItem_idx].strName,
                                        mTmp.strName, MAX_PROP_LEN);
                                app.menuCustomItems[app.nMenuCustomItem_idx].blNeedsTerminal =
                                        mTmp.blNeedsTerminal;

                                app.nMenuCustomItem_idx++;
                                break;
                            default:
                                break;
                        }
                        nIdx++;

                        strTok = strtok(NULL,"|");
                    }
                }
            } else
            // recurse into sub-dirs?
            if (strcmp(strProp, "recurse") == 0
                && strcmp(strVal, "false") == 0
                ) {
                    app.blRecurse = FALSE;
            } else            
            // run command
            if (strcmp(strProp, "runcommand") == 0) {
                if (strlen(strVal) != 0) {
                    strncpy(app.strRunCommand, strVal, MAX_PROP_LEN);
                }
            } else
            // exit command
            if (strcmp(strProp, "exitcommand") == 0) {
                if (strlen(strVal) != 0) {
                    strncpy(app.strExitCommand, strVal, MAX_PROP_LEN);
                }
            } else
            // reboot command
            if (strcmp(strProp, "restartcommand") == 0) {
                if (strlen(strVal) != 0) {
                    strncpy(app.strRebootCommand, strVal, MAX_PROP_LEN);
                }
            } else
            // poweroff command
            if (strcmp(strProp, "poweroffcommand") == 0) {
                if (strlen(strVal) != 0) {
                    strncpy(app.strPoweroffCommand, strVal, MAX_PROP_LEN);
                }
            } else
            // terminal run command
            if (strcmp(strProp, "terminalruncommand") == 0) {
                if (strlen(strVal) != 0) {
                    strncpy(app.strTermRunCommand, strVal, MAX_PROP_LEN);
                }
            }
        }

        // close file
        if (fFile != NULL)
            fclose(fFile);
    }
}


/* parse .desktop files in system, then user 'applications' locations */
void process_desktop_files ()
{
    MenuListItem menuListItem;
    gchar* strFileTemp = NULL;
    gchar strPathFileTemp[MAX_PROP_LEN] = {0};
    GKeyFile* keyfile = g_key_file_new();
    GDir* gdir = NULL;
    gint i = 0;

    FILE* fFile = NULL;

    app.apps_paths_idx = 0;

    // attempt to open first item in strDirs array
    gdir = g_dir_open(app.strAppsPaths[app.apps_paths_idx], 0, NULL);
    if (gdir == NULL) {
        show_error_dialog("No 'Applications' folders found - aborting");
        return;
    }

    // read desktop files, create and store menu items
    while (TRUE) {
        gboolean blNoShow = FALSE;
        fFile = NULL;
        strFileTemp = NULL;

        init_menu_list_item(menuListItem);

        // read next filename in folder
        strFileTemp = (char*)g_dir_read_name(gdir);
        
        if (strFileTemp != NULL) {
            snprintf(strPathFileTemp, MAX_PROP_LEN, "%s/%s", app.strAppsPaths[app.apps_paths_idx],
                     strFileTemp);

            fFile = fopen(strPathFileTemp, "r");
        }

        if (fFile != NULL) {

            // load keyfile stuff from .desktop file
            if (strstr(strFileTemp, ".desktop") != NULL
                && g_key_file_load_from_file(keyfile, strPathFileTemp, G_KEY_FILE_NONE, NULL)
                ) {

                // 'name' key
                gchar* strNameTemp = g_key_file_get_locale_string(keyfile, "Desktop Entry",
                                                                   "Name", NULL, NULL);

                if (strNameTemp != NULL)
                    strncpy(menuListItem.strName, strNameTemp, MAX_PROP_LEN);
                else
                    memset(menuListItem.strName, 0, MAX_PROP_LEN);

                // 'icon' key
                gchar* strIconTemp = g_key_file_get_string(keyfile, "Desktop Entry", "Icon", NULL);

                if (strIconTemp != NULL)
                    strncpy(menuListItem.strIcon, strIconTemp, MAX_PROP_LEN);
                else
                    memset(menuListItem.strIcon, 0, MAX_PROP_LEN);

                // 'exec' key
                gchar* strExecTemp = g_key_file_get_string(keyfile, "Desktop Entry", "Exec", NULL);

                if (strExecTemp != NULL) {
                    strncpy(menuListItem.strExec, strExecTemp, MAX_PROP_LEN);
                    chop_params(menuListItem.strExec);
                } else {
                    memset(menuListItem.strExec, 0, MAX_PROP_LEN);
                }

                // 'categories' key
                gchar* strCatTemp = g_key_file_get_string(keyfile, "Desktop Entry",
                                                          "Categories", NULL);

                if (strCatTemp != NULL)
                    strncpy(menuListItem.strCategory, strCatTemp, MAX_PROP_LEN);
                else
                    memset(menuListItem.strCategory, 0, MAX_PROP_LEN);

                // 'needs terminal' key
                menuListItem.blNeedsTerminal = g_key_file_get_boolean(keyfile, "Desktop Entry",
                                                                      "Terminal", NULL);

                // 'don't display' key
                blNoShow = g_key_file_get_boolean(keyfile, "Desktop Entry", "NoDisplay", NULL);

                // 'show specific desktop environment items' key
                gchar* strDE = g_key_file_get_string(keyfile, "Desktop Entry", "OnlyShowIn", NULL);

                if (strDE != NULL) {
                    
                    strDE = g_ascii_strdown(strDE, -1);

                    // kde
                    if (strcmp(strDE, "kde") == 0 && app.blShowKDE == FALSE)
                        blNoShow = TRUE;
                    else
                    // gnome
                    if (strcmp(strDE, "gnome") == 0 && app.blShowGNOME == FALSE)
                        blNoShow = TRUE;
                    else
                    // mate
                    if (strcmp(strDE, "mate") == 0 && app.blShowMATE == FALSE)
                        blNoShow = TRUE;
                    else
                    // xfce
                    if (strcmp(strDE, "xfce") == 0 && app.blShowXFCE == FALSE)
                        blNoShow = TRUE;
                    else
                    // lxde
                    if (strcmp(strDE, "lxde") == 0 && app.blShowLXDE == FALSE)
                        blNoShow = TRUE;
                    else
                    // cinnamon
                    if (strcmp(strDE, "cinnamon") == 0 && app.blShowCinnamon == FALSE)
                        blNoShow = TRUE;
                    else
                    // unity
                    if (strcmp(strDE, "unity") == 0 && app.blShowUnity == FALSE)
                        blNoShow = TRUE;
                }

                // free keyfile
                if (keyfile != NULL)
                    g_key_file_free(keyfile);

                // clean up and add to menuitemlist
                if (strlen(menuListItem.strName) > 0  && !blNoShow) {
                    if (menuListItem.blNeedsTerminal) {
                        gchar* strETemp = menuListItem.strExec;
                        snprintf(menuListItem.strExec, MAX_PROP_LEN, "%s %s",
                            app.strTermRunCommand, strETemp);
                    }
                    add_to_list(menuListItem);
                }
            } else { 
                // if file is a directory, add it after the last
                // valid item in strDirs array
                if (app.blRecurse == TRUE) {
                    if (g_file_test(strPathFileTemp, G_FILE_TEST_IS_DIR)
                        && strcmp(strFileTemp, ".") != 0
                        && strcmp(strFileTemp, "..") != 0
                    ) {
                        for (i = 0; i < MAX_CUSTOM_LIST_SIZE; i++) {
                            if (app.strAppsPaths[i] != NULL) {
                                if (strcmp(app.strAppsPaths[i], "") == 0) {
                                    strncpy(app.strAppsPaths[i], strPathFileTemp, MAX_PROP_LEN);
                                    break;
                                }
                            }
                        }
                    }
                }
            }  // if is desktop file and keyfile loads...
        } else {    // ...if (fFile != NULL)...
            
            // null file, no more files in current dir (or problem opening)
            
            // close current dir, if open
            if (gdir != NULL)
                g_dir_close(gdir);

            // switch to next dir in array and start again
            app.apps_paths_idx++;
            
            if (app.strAppsPaths[app.apps_paths_idx] != NULL
                && strcmp(app.strAppsPaths[app.apps_paths_idx], "") != 0
                ) {
                gdir = g_dir_open(app.strAppsPaths[app.apps_paths_idx], 0, NULL);            
                // finish up, no more dirs to process
                if (gdir == NULL) {
                    break;
                }
            } else {
                break;
            }
        }    // ...if (fFile != NULL)...

        // close file
        if (fFile != NULL)
            fclose(fFile);
    }
}


/* start building the gtkmenu from menulistitems */
void build_menu ()
{
    GtkWidget* menTemp = NULL;
    GtkWidget* menu_item = NULL;
    char strCategoryTemp[MAX_PROP_LEN] = {0};
    gint i = 0;

    GtkWidget* menPref = NULL;
    GtkWidget* menAcc = NULL;
    GtkWidget* menDevel = NULL;
    GtkWidget* menEdu = NULL;
    GtkWidget* menGame = NULL;
    GtkWidget* menGraph = NULL;
    GtkWidget* menInter = NULL;
    GtkWidget* menMulti = NULL;
    GtkWidget* menOffic = NULL;
    GtkWidget* menOth = NULL;
    GtkWidget* menSci = NULL;
    GtkWidget* menSys = NULL;
    GtkWidget* menHam = NULL;

    // create menu and set up signals
    app.main_menu = gtk_menu_new ();
    gtk_menu_set_screen(GTK_MENU(app.main_menu), gdk_screen_get_default());

    // add custom items first
    for (i = 0; i < app.nMenuCustomItem_idx; i++) {
       // add custom item to menu
        menu_item = gtk_image_menu_item_new_with_label(app.menuCustomItems[i].strName);
        set_item_icon(menu_item, app.menuCustomItems[i].strIcon);
        g_signal_connect(GTK_WIDGET(menu_item), "activate", G_CALLBACK(menuitem_click_handler),
                         app.menuCustomItems[i].strExec);
        gtk_menu_shell_append(GTK_MENU_SHELL(app.main_menu), menu_item);
    }

    // collate by standard categories
    for (i = 0; i < app.menuListItems_idx; i++) {

        strncpy(strCategoryTemp, app.menuListItems[i].strCategory, MAX_PROP_LEN);

        // settings / preferences
        if (strstr(strCategoryTemp, "Settings")) {
            if (menPref == NULL) menPref = gtk_menu_new();
            menTemp = menPref;
        } else
        // ham radio
        if (strstr(strCategoryTemp, "HamRadio"))
        {
            if (menHam == NULL) menHam = gtk_menu_new();
            menTemp = menHam;
        } else
        // science
        if (strstr(strCategoryTemp, "Science"))
        {
            if (menSci == NULL) menSci = gtk_menu_new();
            menTemp = menSci;
        } else
        // accessories / utility
        if (strstr(strCategoryTemp, "Accessories") ||
            strstr(strCategoryTemp, "Utility")
            ) {
            if (menAcc == NULL) menAcc = gtk_menu_new();
            menTemp = menAcc;
        } else
        // development / programming
        if (strstr(strCategoryTemp, "Development")) {
            if (menDevel == NULL) menDevel = gtk_menu_new();
            menTemp = menDevel;
        } else
        // education
        if (strstr(strCategoryTemp, "Education")) {
            if (menEdu == NULL) menEdu = gtk_menu_new();
            menTemp = menEdu;
        } else
        // games
        if (strstr(strCategoryTemp, "Game")) {
            if (menGame == NULL) menGame = gtk_menu_new();
            menTemp = menGame;
        } else
        // graphics
        if (strstr(strCategoryTemp, "Graphics")) {
            if (menGraph == NULL) menGraph = gtk_menu_new();
            menTemp = menGraph;
        } else
        // internet / network
        if (strstr(strCategoryTemp, "Internet") ||
            strstr(strCategoryTemp, "WebBrowser") ||
            strstr(strCategoryTemp, "Network")
            ) {
            if (menInter == NULL) menInter = gtk_menu_new();
            menTemp = menInter;
        } else
        // multimedia / sound/video
        if (strstr(strCategoryTemp, "Audio") ||
            strstr(strCategoryTemp, "AudioVideo") ||
            strstr(strCategoryTemp, "Multimedia") ||
            strstr(strCategoryTemp, "Video")
            ) {
            if (menMulti == NULL) menMulti = gtk_menu_new();
            menTemp = menMulti;
        } else
        // office / productivity
        if (strstr(strCategoryTemp, "Office")  ||
            strstr(strCategoryTemp, "Productivity")  ||
            strstr(strCategoryTemp, "WordProcessor")
            ) {
            if (menOffic == NULL) menOffic = gtk_menu_new();
            menTemp = menOffic;
        } else
        // system
        if (strstr(strCategoryTemp, "System")) {
            if (menSys == NULL) menSys = gtk_menu_new();
            menTemp = menSys;
        } else {
            if (menOth == NULL) menOth = gtk_menu_new();
            menTemp = menOth;
        }

       // add item to correct category menu
        menu_item = gtk_image_menu_item_new_with_label(app.menuListItems[i].strName);
        set_item_icon(menu_item, app.menuListItems[i].strIcon);
        g_signal_connect(GTK_WIDGET(menu_item), "activate", G_CALLBACK(menuitem_click_handler),
            app.menuListItems[i].strExec);
        gtk_menu_shell_append(GTK_MENU_SHELL(menTemp), menu_item);
    }

    // add separator
    gtk_menu_shell_append(GTK_MENU_SHELL(app.main_menu), gtk_separator_menu_item_new());

    // prefs used
    if (menPref != NULL) {
        GtkWidget* catPref = gtk_image_menu_item_new_with_label("Preferences");
        gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(catPref),
            GTK_WIDGET(gtk_image_new_from_icon_name("preferences-desktop", GTK_ICON_SIZE_MENU)));
        gtk_menu_item_set_submenu(GTK_MENU_ITEM(catPref), menPref);
        gtk_menu_shell_append(GTK_MENU_SHELL(app.main_menu), catPref);
        gtk_menu_shell_append(GTK_MENU_SHELL(app.main_menu), gtk_separator_menu_item_new());
    }

    // acc used
    if (menAcc != NULL) {
        GtkWidget* catAcc = gtk_image_menu_item_new_with_label("Accessories");
        gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(catAcc),
            GTK_WIDGET(gtk_image_new_from_icon_name("applications-accessories", GTK_ICON_SIZE_MENU)));
        gtk_menu_item_set_submenu(GTK_MENU_ITEM(catAcc), menAcc);
        gtk_menu_shell_append(GTK_MENU_SHELL(app.main_menu), catAcc);
    }

    // devel used
    if (menDevel != NULL) {
        GtkWidget* catDevel = gtk_image_menu_item_new_with_label("Development");
        gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(catDevel),
            GTK_WIDGET(gtk_image_new_from_icon_name("applications-development", GTK_ICON_SIZE_MENU)));
        gtk_menu_item_set_submenu(GTK_MENU_ITEM(catDevel), menDevel);
        gtk_menu_shell_append(GTK_MENU_SHELL(app.main_menu), catDevel);
    }

    // edu used
    if (menEdu != NULL) {
        GtkWidget* catEdu = gtk_image_menu_item_new_with_label("Education");
        gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(catEdu),
            GTK_WIDGET(gtk_image_new_from_icon_name("applications-science", GTK_ICON_SIZE_MENU)));
        gtk_menu_item_set_submenu(GTK_MENU_ITEM(catEdu), menEdu);
        gtk_menu_shell_append(GTK_MENU_SHELL(app.main_menu), catEdu);
    }

    // game used
    if (menGame != NULL) {
        GtkWidget* catGame = gtk_image_menu_item_new_with_label("Games");
        gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(catGame),
            GTK_WIDGET(gtk_image_new_from_icon_name("applications-games", GTK_ICON_SIZE_MENU)));
        gtk_menu_item_set_submenu(GTK_MENU_ITEM(catGame), menGame);
        gtk_menu_shell_append(GTK_MENU_SHELL(app.main_menu), catGame);
    }

    // graph used
    if (menGraph != NULL) {
        GtkWidget* catGraph = gtk_image_menu_item_new_with_label("Graphics");
        gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(catGraph),
            GTK_WIDGET(gtk_image_new_from_icon_name("applications-graphics", GTK_ICON_SIZE_MENU)));
        gtk_menu_item_set_submenu(GTK_MENU_ITEM(catGraph), menGraph);
        gtk_menu_shell_append(GTK_MENU_SHELL(app.main_menu), catGraph);
    }

    // inter used
    if (menInter != NULL) {
        GtkWidget* catInter = gtk_image_menu_item_new_with_label("Network");
        gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(catInter),
            GTK_WIDGET(gtk_image_new_from_icon_name("applications-internet", GTK_ICON_SIZE_MENU)));
        gtk_menu_item_set_submenu(GTK_MENU_ITEM(catInter), menInter);
        gtk_menu_shell_append(GTK_MENU_SHELL(app.main_menu), catInter);
    }

    // ham used
    if (menHam != NULL) {
        GtkWidget* catHam = gtk_image_menu_item_new_with_label("Ham Radio");
        gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(catHam),
            GTK_WIDGET(gtk_image_new_from_icon_name("applications-other", GTK_ICON_SIZE_MENU)));
        gtk_menu_item_set_submenu(GTK_MENU_ITEM(catHam), menHam);
        gtk_menu_shell_append(GTK_MENU_SHELL(app.main_menu), catHam);
    }

    // multi used
    if (menMulti != NULL) {
        GtkWidget* catMulti = gtk_image_menu_item_new_with_label("Multimedia");
        gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(catMulti),
            GTK_WIDGET(gtk_image_new_from_icon_name("applications-multimedia", GTK_ICON_SIZE_MENU)));
        gtk_menu_item_set_submenu(GTK_MENU_ITEM(catMulti), menMulti);
        gtk_menu_shell_append(GTK_MENU_SHELL(app.main_menu), catMulti);
    }

    // office used
    if (menOffic != NULL) {
        GtkWidget* catOffic = gtk_image_menu_item_new_with_label("Office");
        gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(catOffic),
            GTK_WIDGET(gtk_image_new_from_icon_name("applications-office", GTK_ICON_SIZE_MENU)));
        gtk_menu_item_set_submenu(GTK_MENU_ITEM(catOffic), menOffic);
        gtk_menu_shell_append(GTK_MENU_SHELL(app.main_menu), catOffic);
    }

    // other used
    if (menOth != NULL) {
        GtkWidget* catOth = gtk_image_menu_item_new_with_label("Other");
        gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(catOth),
            GTK_WIDGET(gtk_image_new_from_icon_name("applications-other", GTK_ICON_SIZE_MENU)));
        gtk_menu_item_set_submenu(GTK_MENU_ITEM(catOth), menOth);
        gtk_menu_shell_append(GTK_MENU_SHELL(app.main_menu), catOth);
    }

    // science used
    if (menSci != NULL) {
        GtkWidget* catSci = gtk_image_menu_item_new_with_label("Science");
        gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(catSci),
            GTK_WIDGET(gtk_image_new_from_icon_name("applications-science", GTK_ICON_SIZE_MENU)));
        gtk_menu_item_set_submenu(GTK_MENU_ITEM(catSci), menSci);
        gtk_menu_shell_append(GTK_MENU_SHELL(app.main_menu), catSci);
    }

    // sys used
    if (menSys != NULL) {
        GtkWidget* catSys = gtk_image_menu_item_new_with_label("System");
        gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(catSys),
            GTK_WIDGET(gtk_image_new_from_icon_name("applications-system", GTK_ICON_SIZE_MENU)));
        gtk_menu_item_set_submenu(GTK_MENU_ITEM(catSys), menSys);
        gtk_menu_shell_append(GTK_MENU_SHELL(app.main_menu), catSys);
    }

    // append about item
    gtk_menu_shell_append(GTK_MENU_SHELL(app.main_menu), gtk_separator_menu_item_new());

    // about
    menu_item = gtk_image_menu_item_new_with_label("About...");
    set_item_icon(menu_item, "gtk-about");
    g_signal_connect(GTK_WIDGET(menu_item), "activate", G_CALLBACK(menuitem_click_handler),
                     "...about...");
    gtk_menu_shell_append(GTK_MENU_SHELL(app.main_menu), menu_item);

    /* append command items */
    gtk_menu_shell_append(GTK_MENU_SHELL(app.main_menu), gtk_separator_menu_item_new());

    // run
    menu_item = gtk_menu_item_new_with_label("Run...");
    g_signal_connect(GTK_WIDGET(menu_item), "activate", G_CALLBACK(menuitem_click_handler),
        app.strRunCommand);
    gtk_menu_shell_append(GTK_MENU_SHELL(app.main_menu), menu_item);

    // exit
    gtk_menu_shell_append(GTK_MENU_SHELL(app.main_menu), gtk_separator_menu_item_new());

    menu_item = gtk_image_menu_item_new_with_label("Exit");
    set_item_icon(menu_item, "system-log-out");
    g_signal_connect(GTK_WIDGET(menu_item), "activate", G_CALLBACK(menuitem_click_handler),
        app.strExitCommand);
    gtk_menu_shell_append(GTK_MENU_SHELL(app.main_menu), menu_item);

    // reboot
    menu_item = gtk_image_menu_item_new_with_label("Reboot");
    set_item_icon(menu_item, "view-refresh");
    g_signal_connect(GTK_WIDGET(menu_item), "activate", G_CALLBACK(menuitem_click_handler),
        app.strRebootCommand);
    gtk_menu_shell_append(GTK_MENU_SHELL(app.main_menu), menu_item);

    // shutdown
    menu_item = gtk_image_menu_item_new_with_label("Shutdown");
    set_item_icon(menu_item, "system-shutdown");
    g_signal_connect(GTK_WIDGET(menu_item), "activate", G_CALLBACK(menuitem_click_handler),
        app.strPoweroffCommand);
    gtk_menu_shell_append(GTK_MENU_SHELL(app.main_menu), menu_item);
}


/* display the menu */
gboolean popup_menu ()
{
    // set up callbacks
    g_signal_connect(GTK_WIDGET(app.main_menu), "size-allocate", G_CALLBACK(get_menu_size), NULL);
    g_signal_connect(GTK_MENU_SHELL(app.main_menu), "deactivate", G_CALLBACK(gtk_main_quit), NULL);
    g_signal_connect(GTK_MENU_SHELL(app.main_menu), "cancel", G_CALLBACK(gtk_main_quit), NULL);

    // set menu to visible
    gtk_widget_show_all(GTK_WIDGET(app.main_menu));

    // show menu
    gtk_menu_popup(GTK_MENU(app.main_menu), NULL, NULL, menu_positioner, NULL, 0, 0);
    gtk_menu_reposition(GTK_MENU(app.main_menu));

    return FALSE;
}


/* kill app after a while */
gboolean times_up ()
{
    exit(0);
    return FALSE;
}


/* delayed start */
gboolean delayed_run ()
{
    process_desktop_files();
    sort_list();
    build_menu();
    popup_menu();

    return FALSE;
}


/* Main program */
int main (int argc, char* argv[])
{
    gtk_init(&argc, &argv);

    // app timeout
    g_timeout_add(120000, times_up, NULL);

    init_app();
    process_config_file();

    if (app.nDelayTime < 100)
        app.nDelayTime = 100;

    // delay opening of menu
    g_timeout_add(app.nDelayTime, delayed_run, NULL);

    gtk_main();

    return 0;
}
