/*
 * gtkmenu-standalone
 * ==================
 * (C) Will Brokenbourgh
 * will.brokenbourgh2877@gmail.com
 * http://www.pismotek.com/brainout/
 * - - - -
 * Thank you Jesus!  To God be the glory!  C programming is fun!! :-D
 * - - - -
 *
 * Copyright (C) 2014 - 2019 Will Brokenbourgh - will.brokenbourgh2877@gmail.com
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 */


/* includes */
#include <stdlib.h>
#include <string.h>
#include <gtk/gtk.h>
#include <glib/gstdio.h>

/* app definitions */
#define APP_VERSION      "0.5.1"
#define APP_NAME         "GTK Menu - Standalone"
#define APP_COPYRIGHT    "(C) 2014 - 2019 Will Brokenbourgh"
#define APP_URL          "https://www.pismotek.com/brainout/"
#define APP_ICON         "gtk-about"

/* size definitions */
#define MAX_PROP_LEN          1024
#define MAX_READ_LINE_LEN     5000
#define MAX_LIST_SIZE        32766
#define MAX_CUSTOM_LIST_SIZE   100


/* menulistitem declaration */
typedef struct mli {
    gchar str_category[MAX_PROP_LEN];
    gchar str_exec[MAX_PROP_LEN];
    gchar str_icon[MAX_PROP_LEN];
    gchar str_name[MAX_PROP_LEN];

    gboolean needs_terminal;
} MenuListItem;

/* application globals */
struct appv {
    gchar str_apps_paths[MAX_CUSTOM_LIST_SIZE][MAX_PROP_LEN];
    gint apps_paths_idx;

    gchar str_pix_paths[MAX_CUSTOM_LIST_SIZE][MAX_PROP_LEN];
    gint pix_paths_idx;

    MenuListItem menu_list_items[MAX_LIST_SIZE];
    gint menu_list_items_idx;

    gchar str_term_run_cmd[MAX_PROP_LEN];

    gchar str_run_cmd[MAX_PROP_LEN];
    gchar str_exit_cmd[MAX_PROP_LEN];
    gchar str_reboot_cmd[MAX_PROP_LEN];
    gchar str_poweroff_cmd[MAX_PROP_LEN];

    gint left_pos;
    gint top_pos;
    gint dock_height;
    gint menu_height;
    gint delay_time;

    gboolean show_gnome_items;
    gboolean show_lxde_items;
    gboolean show_kde_items;
    gboolean show_mate_items;
    gboolean show_xfce_items;
    gboolean show_cinnamon_items;
    gboolean show_unity_items;

    gboolean recurse_dirs;
    gboolean show_at_mouse_pos;

    MenuListItem menu_custom_items[MAX_CUSTOM_LIST_SIZE];
    gint menu_custom_items_idx;

    GtkWidget* main_menu;

} app;


/* ###### functions ############### */

/* check and add valid user, sys and pixmap paths */
void add_user_and_system_paths ()
{
    // add user's '$HOME/.local/share/applications' path, if it exists
    gchar* str_user_apps = g_build_path("", g_get_user_data_dir(), "/applications", NULL);

    if (str_user_apps != NULL && g_dir_open(str_user_apps, 0, NULL))
    {
        strncpy(app.str_apps_paths[app.apps_paths_idx], str_user_apps, MAX_PROP_LEN);
        app.apps_paths_idx++;
    }

    // add system application dirs
    if (g_dir_open("/usr/share/applications", 0, NULL))
    {
        strncpy(app.str_apps_paths[app.apps_paths_idx], "/usr/share/applications", MAX_PROP_LEN);
        app.apps_paths_idx++;
    }

    if (g_dir_open("/usr/local/share/applications", 0, NULL))
    {
        strncpy(app.str_apps_paths[app.apps_paths_idx], "/usr/local/share/applications", MAX_PROP_LEN);
        app.apps_paths_idx++;
    }

    // check and add system pixmap dirs
    if (g_dir_open("/usr/share/pixmaps", 0, NULL))
    {
        strncpy(app.str_pix_paths[app.pix_paths_idx], "/usr/share/pixmaps/", MAX_PROP_LEN);
        app.pix_paths_idx++;
    }

    if (g_dir_open("/usr/local/share/pixmaps", 0, NULL))
    {
        strncpy(app.str_pix_paths[app.pix_paths_idx], "/usr/local/share/pixmaps/", MAX_PROP_LEN);
        app.pix_paths_idx++;
    }
}


/* initialize app variables */
void init_app ()
{
    app.apps_paths_idx = 0;
    app.pix_paths_idx = 0;

    add_user_and_system_paths();

    // set defaults for custom commands
    strncpy(app.str_term_run_cmd, "x-terminal-emulator -e", MAX_PROP_LEN);
    strncpy(app.str_run_cmd, "fbrun -nearmouse", MAX_PROP_LEN);
    strncpy(app.str_exit_cmd, "fluxbox-remote quit", MAX_PROP_LEN);
    strncpy(app.str_reboot_cmd, "sudo reboot", MAX_PROP_LEN);
    strncpy(app.str_poweroff_cmd, "sudo poweroff", MAX_PROP_LEN);

    // desktop environment item prefs
    app.show_gnome_items = TRUE;
    app.show_lxde_items = TRUE;
    app.show_kde_items = TRUE;
    app.show_mate_items = TRUE;
    app.show_xfce_items = TRUE;
    app.show_cinnamon_items = TRUE;
    app.show_unity_items = TRUE;

    // other options
    app.recurse_dirs = TRUE;
    app.show_at_mouse_pos = FALSE;
    app.left_pos = -1;
    app.top_pos = -1;
    app.dock_height = 24;
    app.menu_height = 0;
    app.delay_time = 100;

    app.menu_custom_items_idx = 0;
}


/* return property and value from input */
void get_prop_and_val (gchar* strIn, gchar* str_prop, gchar* str_val)
{
    gchar str_property[MAX_PROP_LEN] = {0};
    gchar str_value[MAX_PROP_LEN] = {0};

    gint idx = 0;
    gint i = 0;

    // property
    for (i = 0; i < MAX_PROP_LEN; i++)
    {
        if (strIn[i] != '=' && strIn[i] != '\0')
            str_property[idx++] = strIn[i];
        else
        {
            str_property[idx++] = '\0';
            strncpy(str_prop, str_property, MAX_PROP_LEN);
            break;
        }
    }

    // value
    idx = 0;
    i++;

    for (; i < MAX_PROP_LEN; i++)
    {
        if (strIn[i] != 0)
            str_value[idx++] = strIn[i];
        else
        {
            str_value[idx++] = '\0';
            strncpy(str_val, str_value, MAX_PROP_LEN);
            return;
        }
    }
}


/* remove menu item exec parameters such as: %U, etc */
void chop_params (gchar* str_base)
{
    if (str_base == NULL)
        return;

    gint base_len = strlen(str_base);

    if (base_len < 1)
        return;

    // find the first '%', if any
    gint pos = strcspn(str_base, "\%");

    if (pos == base_len)
        return;

    // set the first '%' to a null character
    str_base[pos] = '\0';
}


/* clears/inits menulistitem */
void init_menu_list_item (MenuListItem menu_list_item)
{
    menu_list_item.str_category[0] = '\0';
    menu_list_item.str_exec[0] = '\0';
    menu_list_item.str_icon[0] = '\0';
    menu_list_item.str_name[0] = '\0';
    menu_list_item.needs_terminal = FALSE;
}


/* add menu_list_item to menu_list_items array */
void add_to_list (MenuListItem menu_list_item)
{
    strncpy(app.menu_list_items[app.menu_list_items_idx].str_category, menu_list_item.str_category,
            MAX_PROP_LEN - 1);
    strncpy(app.menu_list_items[app.menu_list_items_idx].str_exec, menu_list_item.str_exec, MAX_PROP_LEN - 1);
    strncpy(app.menu_list_items[app.menu_list_items_idx].str_icon, menu_list_item.str_icon, MAX_PROP_LEN - 1);
    strncpy(app.menu_list_items[app.menu_list_items_idx].str_name, menu_list_item.str_name, MAX_PROP_LEN - 1);
    app.menu_list_items[app.menu_list_items_idx].needs_terminal = menu_list_item.needs_terminal;

    app.menu_list_items_idx++;
}


/* qsort compare function - sort by menuitem name */
int qsort_compare (const void* item_a, const void* item_b)
{
    return strcmp(((MenuListItem*)item_a)->str_name, ((MenuListItem*)item_b)->str_name);
}


/* sort menuitemlist */
void sort_list ()
{
    qsort(app.menu_list_items, app.menu_list_items_idx, sizeof(MenuListItem), qsort_compare);
}


/* set menuitem's icon depending on str_icon contents */
void set_item_icon(GtkWidget* menuitem, const gchar* str_icon)
{
    gboolean has_path = FALSE;
    gboolean has_extension = FALSE;
    GtkWidget* img = NULL;
    GdkPixbuf* pbi = NULL;
    gchar str_pix_path[MAX_PROP_LEN] = {0};
    GError* gerr = NULL;
    gint i = 0;
    gchar* str_temp = NULL;

    if (menuitem == NULL || str_icon == NULL)
        return;

    // normalize temp string to lower-case
    str_temp = g_ascii_strdown(str_icon, -1);

    // check for absolute path
    if (strchr(str_temp, '/') != NULL)
        has_path = TRUE;

    // check for extension (.png, .svg, etc)
    if (strstr(str_temp, ".png") != NULL ||
        strstr(str_icon, ".svg") != NULL ||
        strstr(str_icon, ".xpm") != NULL
      )
        has_extension = TRUE;

    // no path or extension
    if (has_path == FALSE && has_extension == FALSE)
    {
        gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(menuitem),
                                      GTK_WIDGET(gtk_image_new_from_icon_name(str_icon,
                                      GTK_ICON_SIZE_MENU)));
        return;
    }

    // has absolute path
    if (has_path == TRUE)
    {
        pbi = gdk_pixbuf_new_from_file_at_scale (str_icon, 16, 16, TRUE, &gerr);

        if (gerr != NULL)
            img = gtk_image_new_from_icon_name("image-missing", GTK_ICON_SIZE_MENU);
        else
            img = gtk_image_new_from_pixbuf(pbi);

        if (GTK_IS_WIDGET(img))
        {
            gtk_widget_set_size_request(GTK_WIDGET(img), 16, 16);
            gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(menuitem), GTK_WIDGET(img));
        }

        return;
    }

    // has extension but NO absolute path
    if (has_extension == TRUE && has_path == FALSE)
    {
        for (i = 0; i < app.pix_paths_idx; i++)
        {
            strncpy(str_pix_path, app.str_pix_paths[i], MAX_PROP_LEN);
            strncat(str_pix_path, "/", MAX_PROP_LEN);
            strncat(str_pix_path, str_icon, MAX_PROP_LEN);

            pbi = gdk_pixbuf_new_from_file_at_scale(str_pix_path, 16, 16, TRUE, &gerr);

            if (gerr != NULL)
            {
                memset(str_pix_path, 0, sizeof(str_pix_path));
                strncpy(str_pix_path, str_icon, MAX_PROP_LEN);
                gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(menuitem),
                                              GTK_WIDGET(gtk_image_new_from_icon_name(str_pix_path,
                                              GTK_ICON_SIZE_MENU)));
            }
            else
                img = gtk_image_new_from_pixbuf(pbi);

            if (GTK_IS_WIDGET(img))
            {
                gtk_widget_set_size_request(GTK_WIDGET(img), 16, 16);
                gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(menuitem), GTK_WIDGET(img));
            }
        }
        return;
    }
}


/* get menu height for positioning calculations */
void get_menu_size (GtkWidget* widget, GtkAllocation* allocation)
{
    app.menu_height = allocation->height;
}


/* position menu properly */
void menu_positioner (GtkMenu* menu, gint* xx, gint* yy, gboolean* push_in, gpointer user_data)
{
    gint screen_height = gdk_screen_get_height(gdk_screen_get_default());

    if (app.left_pos == -1)
        app.left_pos = 0;

    if (app.top_pos == -1)
        app.top_pos = 0;

    *push_in = TRUE;

    *xx = app.left_pos;

    if (app.menu_height == 0)
        *yy = 0;
    else
        *yy = ((screen_height - app.menu_height) - app.dock_height) - app.top_pos;
}


/* display About box */
void show_about_dialog ()
{
    GtkWidget* about_dialog = NULL;
    about_dialog = gtk_about_dialog_new();
    gtk_about_dialog_set_program_name(GTK_ABOUT_DIALOG(about_dialog), APP_NAME);
    gtk_about_dialog_set_copyright(GTK_ABOUT_DIALOG(about_dialog), APP_COPYRIGHT);
    gtk_about_dialog_set_version(GTK_ABOUT_DIALOG(about_dialog), APP_VERSION);
    gtk_about_dialog_set_comments(GTK_ABOUT_DIALOG(about_dialog), APP_URL);
    gtk_about_dialog_set_logo_icon_name(GTK_ABOUT_DIALOG(about_dialog), APP_ICON);

    gtk_dialog_run(GTK_DIALOG(about_dialog));
}


/* display Error dialog */
void show_error_dialog (gchar* str_error_in)
{
    GtkWidget* error_dialog = NULL;

    if (str_error_in == NULL)
        return;

    error_dialog = gtk_message_dialog_new(NULL, GTK_DIALOG_MODAL, GTK_MESSAGE_ERROR,
        GTK_BUTTONS_OK, str_error_in);

    gtk_window_set_title(GTK_WINDOW(error_dialog), "Gtk Menu - Error");

    gtk_dialog_run(GTK_DIALOG(error_dialog));
}


/* handle menuitem click */
void menuitem_click_handler (GtkMenuItem* menuitem, gpointer user_data)
{
    GError* gerr = NULL;

    if (user_data == NULL)
        return;

    g_chdir(getenv("HOME"));

    if (strcmp((gchar*)user_data, "...about...") == 0)
        // show About dialog
        show_about_dialog();
    else
    {
        // execute menuitem command
        g_spawn_command_line_async((gchar*)user_data, &gerr);

        if (gerr != NULL)
            show_error_dialog(gerr->message);
    }

    gtk_main_quit();
}


/* parse config file or set defaults if none */
void process_config_file ()
{
    gchar char_process = 0;
    gint pos = -1;
    gchar str_process[MAX_READ_LINE_LEN];
    gchar str_prop[MAX_PROP_LEN];
    gchar str_val[MAX_PROP_LEN];

    FILE* file_handle = NULL;

    // attempt to open ~/.config/gtkmenu-standalone/config file
    file_handle = fopen(g_build_filename("", g_get_user_config_dir(),
        "gtkmenu-standalone/config", NULL) , "r");

    if (file_handle == NULL)
    {
        // couldn't open config file, just use defaults
        printf("gtkmenu-standalone: Couldn't open config file.  Will use defaults.\n");
        return;
    }
    else
    {
        // opened config file, let's process it
        while (feof(file_handle) == FALSE)
        {
            pos = -1;
            char_process = 0;

            // read characters from file, stop at eol or eof
            // so one (file) line is created
            while (char_process != EOF)
            {
                char_process = fgetc(file_handle);

                pos++;

                if (char_process == '\n' ||
                    char_process == '\r' ||
                    char_process == 0 ||
                    char_process == EOF ||
                    pos > (MAX_READ_LINE_LEN - 2)
                    )
                {
                        str_process[pos] = '\0';
                        break;
                }

                str_process[pos] = char_process;
            }

            // parse line and set 'property' and 'value'
            get_prop_and_val(str_process, str_prop, str_val);

            // normalize property name to lower-case
            strncpy(str_prop, g_ascii_strdown(str_prop, -1), MAX_PROP_LEN);

            // left positioning
            if (strcmp(str_prop, "left") == 0)
                app.left_pos = atoi(str_val);
            else
            // top_pos positioning
            if (strcmp(str_prop, "top_pos") == 0)
                app.top_pos = atoi(str_val);
            else
            // users dock size
            if (strcmp(str_prop, "docksize") == 0)
                app.dock_height = atoi(str_val);
            else
            // menu pop-up delay time
            if (strcmp(str_prop, "delay") == 0)
                app.delay_time = atoi(str_val);
            else
            // show GNOME items?
            if (strcmp(str_prop, "gnome") == 0
                && strcmp(str_val, "false") == 0
               )
                app.show_gnome_items = FALSE;
            else
            // show KDE items?
            if (strcmp(str_prop, "kde") == 0
                && strcmp(str_val, "false") == 0
               )
                app.show_kde_items = FALSE;
            else
            // show LXDE items?
            if (strcmp(str_prop, "lxde") == 0
                && strcmp(str_val, "false") == 0
               )
                app.show_lxde_items = FALSE;
            else
            // show MATE items?
            if (strcmp(str_prop, "mate") == 0
                && strcmp(str_val, "false") == 0
               )
                app.show_mate_items = FALSE;
            else
            // show XFCE items?
            if (strcmp(str_prop, "xfce") == 0
                && strcmp(str_val, "false") == 0
               )
                app.show_xfce_items = FALSE;
            else
            // show Cinnamon items?
            if (strcmp(str_prop, "cinnamon") == 0
                && strcmp(str_val, "false") == 0
               )
                app.show_cinnamon_items = FALSE;
            else
            // show Unity items?
            if (strcmp(str_prop, "unity") == 0
                && strcmp(str_val, "false") == 0
               )
                app.show_unity_items = FALSE;
            else
            // custom menu items
            if (strcmp(str_prop, "custom") == 0)
            {
                gchar* str_token = NULL;
                int idx = 0;
                MenuListItem m_tmp;

                if (str_val != NULL)
                {
                    init_menu_list_item(m_tmp);

                    strncpy(m_tmp.str_category, "CustomItems", MAX_PROP_LEN);

                    str_token = strtok(str_val, "|");

                    while (str_token != NULL)
                    {
                        switch (idx)
                        {
                            case 0:
                                strncpy(m_tmp.str_name, str_token, MAX_PROP_LEN);
                                break;
                            case 1:
                                strncpy(m_tmp.str_exec, str_token, MAX_PROP_LEN);
                                break;
                            case 2:
                                strncpy(m_tmp.str_icon, str_token, MAX_PROP_LEN);

                                strncpy(app.menu_custom_items[app.menu_custom_items_idx].str_category,
                                        m_tmp.str_category, MAX_PROP_LEN);
                                strncpy(app.menu_custom_items[app.menu_custom_items_idx].str_exec,
                                        m_tmp.str_exec, MAX_PROP_LEN);
                                strncpy(app.menu_custom_items[app.menu_custom_items_idx].str_icon,
                                        m_tmp.str_icon, MAX_PROP_LEN);
                                strncpy(app.menu_custom_items[app.menu_custom_items_idx].str_name,
                                        m_tmp.str_name, MAX_PROP_LEN);
                                app.menu_custom_items[app.menu_custom_items_idx].needs_terminal =
                                        m_tmp.needs_terminal;

                                app.menu_custom_items_idx++;
                                break;
                            default:
                                break;
                        }
                        idx++;

                        str_token = strtok(NULL,"|");
                    }
                }
            }
            else
            // recurse into sub-dirs?
            if (strcmp(str_prop, "recurse") == 0
                && strcmp(str_val, "false") == 0
               )
                app.recurse_dirs = FALSE;
            else
            // show at mouse position?
            if (strcmp(str_prop, "showatmouse") == 0
                && strcmp(str_val, "true") == 0
               )
                app.show_at_mouse_pos = TRUE;
            else
            // run command
            if (strcmp(str_prop, "runcommand") == 0)
            {
                if (strlen(str_val) != 0)
                    strncpy(app.str_run_cmd, str_val, MAX_PROP_LEN);
            }
            else
            // exit command
            if (strcmp(str_prop, "exitcommand") == 0)
            {
                if (strlen(str_val) != 0)
                    strncpy(app.str_exit_cmd, str_val, MAX_PROP_LEN);
            }
            else
            // reboot command
            if (strcmp(str_prop, "restartcommand") == 0)
            {
                if (strlen(str_val) != 0)
                    strncpy(app.str_reboot_cmd, str_val, MAX_PROP_LEN);
            } else
            // poweroff command
            if (strcmp(str_prop, "poweroffcommand") == 0)
            {
                if (strlen(str_val) != 0)
                    strncpy(app.str_poweroff_cmd, str_val, MAX_PROP_LEN);
            }
            else
            // terminal run command
            if (strcmp(str_prop, "terminalruncommand") == 0)
            {
                if (strlen(str_val) != 0)
                    strncpy(app.str_term_run_cmd, str_val, MAX_PROP_LEN);
            }
        }

        // close file
        if (file_handle != NULL)
            fclose(file_handle);
    }
}


/* parse .desktop files in system, then user 'applications' locations */
void process_desktop_files ()
{
    MenuListItem menu_list_item;
    gchar* str_file_temp = NULL;
    gchar str_path_file_temp[MAX_PROP_LEN] = {0};
    GKeyFile* keyfile = NULL; // = g_key_file_new();
    GDir* gdir = NULL;
    gint i = 0;

    FILE* file_handle = NULL;

    app.apps_paths_idx = 0;

    // attempt to open first item in strDirs array
    gdir = g_dir_open(app.str_apps_paths[app.apps_paths_idx], 0, NULL);

    if (gdir == NULL)
    {
        show_error_dialog("No 'Applications' folders found - aborting");
        return;
    }

    // read desktop files, create and store menu items
    while (1)
    {
        gboolean blNoShow = FALSE;
        file_handle = NULL;
        str_file_temp = NULL;

        init_menu_list_item(menu_list_item);

        // read next filename in folder
        str_file_temp = (char*)g_dir_read_name(gdir);

        if (str_file_temp != NULL)
        {
            snprintf(str_path_file_temp, MAX_PROP_LEN, "%s/%s", app.str_apps_paths[app.apps_paths_idx],
                     str_file_temp);

            file_handle = fopen(str_path_file_temp, "r");
        }

        if (file_handle != NULL)
        {
            keyfile = g_key_file_new();

            // load keyfile stuff from .desktop file
            if (strstr(str_file_temp, ".desktop") != NULL
                && g_key_file_load_from_file(keyfile, str_path_file_temp, G_KEY_FILE_NONE, NULL)
               )
            {
                // 'name' key
                gchar* strNameTemp = g_key_file_get_locale_string(keyfile, "Desktop Entry",
                                                                   "Name", NULL, NULL);

                if (strNameTemp != NULL)
                    strncpy(menu_list_item.str_name, strNameTemp, MAX_PROP_LEN);
                else
                    memset(menu_list_item.str_name, 0, MAX_PROP_LEN);

                // 'icon' key
                gchar* strIconTemp = g_key_file_get_string(keyfile, "Desktop Entry", "Icon", NULL);

                if (strIconTemp != NULL)
                    strncpy(menu_list_item.str_icon, strIconTemp, MAX_PROP_LEN);
                else
                    memset(menu_list_item.str_icon, 0, MAX_PROP_LEN);

                // 'exec' key
                gchar* strExecTemp = g_key_file_get_string(keyfile, "Desktop Entry", "Exec", NULL);

                if (strExecTemp != NULL)
                {
                    strncpy(menu_list_item.str_exec, strExecTemp, MAX_PROP_LEN);
                    chop_params(menu_list_item.str_exec);
                }
                else
                    memset(menu_list_item.str_exec, 0, MAX_PROP_LEN);

                // 'categories' key
                gchar* strCatTemp = g_key_file_get_string(keyfile, "Desktop Entry",
                                                          "Categories", NULL);

                if (strCatTemp != NULL)
                    strncpy(menu_list_item.str_category, strCatTemp, MAX_PROP_LEN);
                else
                    memset(menu_list_item.str_category, 0, MAX_PROP_LEN);

                // 'needs terminal' key
                menu_list_item.needs_terminal = g_key_file_get_boolean(keyfile, "Desktop Entry",
                                                                      "Terminal", NULL);

                // 'don't display' key
                blNoShow = g_key_file_get_boolean(keyfile, "Desktop Entry", "NoDisplay", NULL);

                // 'show specific desktop environment items' key
                gchar* strDE = g_key_file_get_string(keyfile, "Desktop Entry", "OnlyShowIn", NULL);

                if (strDE != NULL)
                {
                    strDE = g_ascii_strdown(strDE, -1);

                    // kde
                    if (strcmp(strDE, "kde") == 0 && app.show_kde_items == FALSE)
                        blNoShow = TRUE;
                    else
                    // gnome
                    if (strcmp(strDE, "gnome") == 0 && app.show_gnome_items == FALSE)
                        blNoShow = TRUE;
                    else
                    // mate
                    if (strcmp(strDE, "mate") == 0 && app.show_mate_items == FALSE)
                        blNoShow = TRUE;
                    else
                    // xfce
                    if (strcmp(strDE, "xfce") == 0 && app.show_xfce_items == FALSE)
                        blNoShow = TRUE;
                    else
                    // lxde
                    if (strcmp(strDE, "lxde") == 0 && app.show_lxde_items == FALSE)
                        blNoShow = TRUE;
                    else
                    // cinnamon
                    if (strcmp(strDE, "cinnamon") == 0 && app.show_cinnamon_items == FALSE)
                        blNoShow = TRUE;
                    else
                    // unity
                    if (strcmp(strDE, "unity") == 0 && app.show_unity_items == FALSE)
                        blNoShow = TRUE;
                }

                // free keyfile
                if (keyfile != NULL)
                    g_key_file_free(keyfile);

                // clean up and add to menuitemlist
                if (strlen(menu_list_item.str_name) > 0  && !blNoShow)
                {
                    if (menu_list_item.needs_terminal)
                    {
                        gchar* strETemp = menu_list_item.str_exec;
                        snprintf(menu_list_item.str_exec, MAX_PROP_LEN, "%s %s",
                            app.str_term_run_cmd, strETemp);
                    }
                    add_to_list(menu_list_item);
                }
            }
            else
            {
                // if file is a directory, add it after the last
                // valid item in strDirs array
                if (app.recurse_dirs == TRUE)
                {
                    if (g_file_test(str_path_file_temp, G_FILE_TEST_IS_DIR)
                        && strcmp(str_file_temp, ".") != 0
                        && strcmp(str_file_temp, "..") != 0
                    )
                    {
                        for (i = 0; i < MAX_CUSTOM_LIST_SIZE; i++)
                        {
                            if (app.str_apps_paths[i] != NULL)
                            {
                                if (strcmp(app.str_apps_paths[i], "") == 0)
                                {
                                    strncpy(app.str_apps_paths[i], str_path_file_temp, MAX_PROP_LEN);
                                    break;
                                }
                            }
                        }
                    }
                }
            }  // if is desktop file and keyfile loads...
        }
        else
        {    // ...if (file_handle != NULL)...
            // null file, no more files in current dir (or problem opening)

            // close current dir, if open
            if (gdir != NULL)
                g_dir_close(gdir);

            // switch to next dir in array and start again
            app.apps_paths_idx++;

            if (app.str_apps_paths[app.apps_paths_idx] != NULL
                && strcmp(app.str_apps_paths[app.apps_paths_idx], "") != 0
                )
            {
                gdir = g_dir_open(app.str_apps_paths[app.apps_paths_idx], 0, NULL);
                // finish up, no more dirs to process
                if (gdir == NULL)
                    break;
            }
            else
                break;
        }    // ...if (file_handle != NULL)...

        // close file
        if (file_handle != NULL)
            fclose(file_handle);
    }
}


/* start building the gtkmenu from menu_list_items */
void build_menu ()
{
    GtkWidget* menu_temp = NULL;
    GtkWidget* menu_item = NULL;
    char str_category_temp[MAX_PROP_LEN] = {0};
    gint i = 0;

    GtkWidget* menu_pref = NULL;
    GtkWidget* menu_acc = NULL;
    GtkWidget* menu_devel = NULL;
    GtkWidget* menu_edu = NULL;
    GtkWidget* menu_game = NULL;
    GtkWidget* menu_graph = NULL;
    GtkWidget* menu_net = NULL;
    GtkWidget* menu_multi = NULL;
    GtkWidget* menu_office = NULL;
    GtkWidget* menu_other = NULL;
    GtkWidget* menu_sci = NULL;
    GtkWidget* menu_sys = NULL;
    GtkWidget* menu_ham = NULL;

    // create menu and set up signals
    app.main_menu = gtk_menu_new ();
    gtk_menu_set_screen(GTK_MENU(app.main_menu), gdk_screen_get_default());

    // add custom items first
    for (i = 0; i < app.menu_custom_items_idx; i++)
    {
       // add custom item to menu
        menu_item = gtk_image_menu_item_new_with_label(app.menu_custom_items[i].str_name);
        set_item_icon(menu_item, app.menu_custom_items[i].str_icon);
        g_signal_connect(GTK_WIDGET(menu_item), "activate", G_CALLBACK(menuitem_click_handler),
                         app.menu_custom_items[i].str_exec);
        gtk_menu_shell_append(GTK_MENU_SHELL(app.main_menu), menu_item);
    }

    // collate by standard categories
    for (i = 0; i < app.menu_list_items_idx; i++)
    {
        strncpy(str_category_temp, app.menu_list_items[i].str_category, MAX_PROP_LEN);

        // settings / preferences
        if (strstr(str_category_temp, "Settings"))
        {
            if (menu_pref == NULL)
                menu_pref = gtk_menu_new();
            menu_temp = menu_pref;
        }
        else
        // ham radio
        if (strstr(str_category_temp, "HamRadio"))
        {
            if (menu_ham == NULL)
                menu_ham = gtk_menu_new();
            menu_temp = menu_ham;
        }
        else
        // science
        if (strstr(str_category_temp, "Science"))
        {
            if (menu_sci == NULL)
                menu_sci = gtk_menu_new();
            menu_temp = menu_sci;
        }
        else
        // accessories / utility
        if (strstr(str_category_temp, "Accessories") ||
            strstr(str_category_temp, "Utility")
            )
        {
            if (menu_acc == NULL)
                menu_acc = gtk_menu_new();
            menu_temp = menu_acc;
        }
        else
        // development / programming
        if (strstr(str_category_temp, "Development"))
        {
            if (menu_devel == NULL)
                menu_devel = gtk_menu_new();
            menu_temp = menu_devel;
        }
        else
        // education
        if (strstr(str_category_temp, "Education"))
        {
            if (menu_edu == NULL)
                menu_edu = gtk_menu_new();
            menu_temp = menu_edu;
        }
        else
        // games
        if (strstr(str_category_temp, "Game"))
        {
            if (menu_game == NULL)
                menu_game = gtk_menu_new();
            menu_temp = menu_game;
        }
        else
        // graphics
        if (strstr(str_category_temp, "Graphics"))
        {
            if (menu_graph == NULL)
                menu_graph = gtk_menu_new();
            menu_temp = menu_graph;
        }
        else
        // internet / network
        if (strstr(str_category_temp, "Internet") ||
            strstr(str_category_temp, "WebBrowser") ||
            strstr(str_category_temp, "Network")
            )
        {
            if (menu_net == NULL)
                menu_net = gtk_menu_new();
            menu_temp = menu_net;
        }
        else
        // multimedia / sound/video
        if (strstr(str_category_temp, "Audio") ||
            strstr(str_category_temp, "AudioVideo") ||
            strstr(str_category_temp, "Multimedia") ||
            strstr(str_category_temp, "Video")
            )
        {
            if (menu_multi == NULL)
                menu_multi = gtk_menu_new();
            menu_temp = menu_multi;
        }
        else
        // office / productivity
        if (strstr(str_category_temp, "Office")  ||
            strstr(str_category_temp, "Productivity")  ||
            strstr(str_category_temp, "WordProcessor")
            )
        {
            if (menu_office == NULL)
                menu_office = gtk_menu_new();
            menu_temp = menu_office;
        }
        else
        // system
        if (strstr(str_category_temp, "System"))
        {
            if (menu_sys == NULL)
                menu_sys = gtk_menu_new();
            menu_temp = menu_sys;
        }
        else
        {
            if (menu_other == NULL)
                menu_other = gtk_menu_new();
            menu_temp = menu_other;
        }

       // add item to correct category menu
        menu_item = gtk_image_menu_item_new_with_label(app.menu_list_items[i].str_name);
        set_item_icon(menu_item, app.menu_list_items[i].str_icon);
        g_signal_connect(GTK_WIDGET(menu_item), "activate", G_CALLBACK(menuitem_click_handler),
            app.menu_list_items[i].str_exec);
        gtk_menu_shell_append(GTK_MENU_SHELL(menu_temp), menu_item);
    }

    // add separator
    gtk_menu_shell_append(GTK_MENU_SHELL(app.main_menu), gtk_separator_menu_item_new());

    // prefs used
    if (menu_pref != NULL)
    {
        GtkWidget* cat_pref = gtk_image_menu_item_new_with_label("Preferences");
        gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(cat_pref),
            GTK_WIDGET(gtk_image_new_from_icon_name("preferences-desktop", GTK_ICON_SIZE_MENU)));
        gtk_menu_item_set_submenu(GTK_MENU_ITEM(cat_pref), menu_pref);
        gtk_menu_shell_append(GTK_MENU_SHELL(app.main_menu), cat_pref);
        gtk_menu_shell_append(GTK_MENU_SHELL(app.main_menu), gtk_separator_menu_item_new());
    }

    // acc used
    if (menu_acc != NULL)
    {
        GtkWidget* cat_acc = gtk_image_menu_item_new_with_label("Accessories");
        gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(cat_acc),
            GTK_WIDGET(gtk_image_new_from_icon_name("applications-accessories", GTK_ICON_SIZE_MENU)));
        gtk_menu_item_set_submenu(GTK_MENU_ITEM(cat_acc), menu_acc);
        gtk_menu_shell_append(GTK_MENU_SHELL(app.main_menu), cat_acc);
    }

    // devel used
    if (menu_devel != NULL)
    {
        GtkWidget* cat_devel = gtk_image_menu_item_new_with_label("Development");
        gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(cat_devel),
            GTK_WIDGET(gtk_image_new_from_icon_name("applications-development", GTK_ICON_SIZE_MENU)));
        gtk_menu_item_set_submenu(GTK_MENU_ITEM(cat_devel), menu_devel);
        gtk_menu_shell_append(GTK_MENU_SHELL(app.main_menu), cat_devel);
    }

    // edu used
    if (menu_edu != NULL)
    {
        GtkWidget* cat_edu = gtk_image_menu_item_new_with_label("Education");
        gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(cat_edu),
            GTK_WIDGET(gtk_image_new_from_icon_name("applications-science", GTK_ICON_SIZE_MENU)));
        gtk_menu_item_set_submenu(GTK_MENU_ITEM(cat_edu), menu_edu);
        gtk_menu_shell_append(GTK_MENU_SHELL(app.main_menu), cat_edu);
    }

    // game used
    if (menu_game != NULL)
    {
        GtkWidget* cat_game = gtk_image_menu_item_new_with_label("Games");
        gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(cat_game),
            GTK_WIDGET(gtk_image_new_from_icon_name("applications-games", GTK_ICON_SIZE_MENU)));
        gtk_menu_item_set_submenu(GTK_MENU_ITEM(cat_game), menu_game);
        gtk_menu_shell_append(GTK_MENU_SHELL(app.main_menu), cat_game);
    }

    // graph used
    if (menu_graph != NULL)
    {
        GtkWidget* cat_graphics = gtk_image_menu_item_new_with_label("Graphics");
        gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(cat_graphics),
            GTK_WIDGET(gtk_image_new_from_icon_name("applications-graphics", GTK_ICON_SIZE_MENU)));
        gtk_menu_item_set_submenu(GTK_MENU_ITEM(cat_graphics), menu_graph);
        gtk_menu_shell_append(GTK_MENU_SHELL(app.main_menu), cat_graphics);
    }

    // net used
    if (menu_net != NULL)
    {
        GtkWidget* cat_network = gtk_image_menu_item_new_with_label("Network");
        gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(cat_network),
            GTK_WIDGET(gtk_image_new_from_icon_name("applications-internet", GTK_ICON_SIZE_MENU)));
        gtk_menu_item_set_submenu(GTK_MENU_ITEM(cat_network), menu_net);
        gtk_menu_shell_append(GTK_MENU_SHELL(app.main_menu), cat_network);
    }

    // ham used
    if (menu_ham != NULL)
    {
        GtkWidget* cat_ham = gtk_image_menu_item_new_with_label("Ham Radio");
        gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(cat_ham),
            GTK_WIDGET(gtk_image_new_from_icon_name("applications-other", GTK_ICON_SIZE_MENU)));
        gtk_menu_item_set_submenu(GTK_MENU_ITEM(cat_ham), menu_ham);
        gtk_menu_shell_append(GTK_MENU_SHELL(app.main_menu), cat_ham);
    }

    // multi used
    if (menu_multi != NULL)
    {
        GtkWidget* cat_multi = gtk_image_menu_item_new_with_label("Multimedia");
        gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(cat_multi),
            GTK_WIDGET(gtk_image_new_from_icon_name("applications-multimedia", GTK_ICON_SIZE_MENU)));
        gtk_menu_item_set_submenu(GTK_MENU_ITEM(cat_multi), menu_multi);
        gtk_menu_shell_append(GTK_MENU_SHELL(app.main_menu), cat_multi);
    }

    // office used
    if (menu_office != NULL)
    {
        GtkWidget* cat_office = gtk_image_menu_item_new_with_label("Office");
        gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(cat_office),
            GTK_WIDGET(gtk_image_new_from_icon_name("applications-office", GTK_ICON_SIZE_MENU)));
        gtk_menu_item_set_submenu(GTK_MENU_ITEM(cat_office), menu_office);
        gtk_menu_shell_append(GTK_MENU_SHELL(app.main_menu), cat_office);
    }

    // other used
    if (menu_other != NULL)
    {
        GtkWidget* cat_other = gtk_image_menu_item_new_with_label("Other");
        gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(cat_other),
            GTK_WIDGET(gtk_image_new_from_icon_name("applications-other", GTK_ICON_SIZE_MENU)));
        gtk_menu_item_set_submenu(GTK_MENU_ITEM(cat_other), menu_other);
        gtk_menu_shell_append(GTK_MENU_SHELL(app.main_menu), cat_other);
    }

    // science used
    if (menu_sci != NULL)
    {
        GtkWidget* cat_sci = gtk_image_menu_item_new_with_label("Science");
        gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(cat_sci),
            GTK_WIDGET(gtk_image_new_from_icon_name("applications-science", GTK_ICON_SIZE_MENU)));
        gtk_menu_item_set_submenu(GTK_MENU_ITEM(cat_sci), menu_sci);
        gtk_menu_shell_append(GTK_MENU_SHELL(app.main_menu), cat_sci);
    }

    // sys used
    if (menu_sys != NULL)
    {
        GtkWidget* cat_sys = gtk_image_menu_item_new_with_label("System");
        gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(cat_sys),
            GTK_WIDGET(gtk_image_new_from_icon_name("applications-system", GTK_ICON_SIZE_MENU)));
        gtk_menu_item_set_submenu(GTK_MENU_ITEM(cat_sys), menu_sys);
        gtk_menu_shell_append(GTK_MENU_SHELL(app.main_menu), cat_sys);
    }

    // append about item
    gtk_menu_shell_append(GTK_MENU_SHELL(app.main_menu), gtk_separator_menu_item_new());

    // about
    menu_item = gtk_image_menu_item_new_with_label("About...");
    set_item_icon(menu_item, "gtk-about");
    g_signal_connect(GTK_WIDGET(menu_item), "activate", G_CALLBACK(menuitem_click_handler),
                     "...about...");
    gtk_menu_shell_append(GTK_MENU_SHELL(app.main_menu), menu_item);

    /* append command items */
    gtk_menu_shell_append(GTK_MENU_SHELL(app.main_menu), gtk_separator_menu_item_new());

    // run
    menu_item = gtk_menu_item_new_with_label("Run...");
    g_signal_connect(GTK_WIDGET(menu_item), "activate", G_CALLBACK(menuitem_click_handler),
        app.str_run_cmd);
    gtk_menu_shell_append(GTK_MENU_SHELL(app.main_menu), menu_item);

    // exit
    gtk_menu_shell_append(GTK_MENU_SHELL(app.main_menu), gtk_separator_menu_item_new());

    menu_item = gtk_image_menu_item_new_with_label("Exit");
    set_item_icon(menu_item, "system-log-out");
    g_signal_connect(GTK_WIDGET(menu_item), "activate", G_CALLBACK(menuitem_click_handler),
        app.str_exit_cmd);
    gtk_menu_shell_append(GTK_MENU_SHELL(app.main_menu), menu_item);

    // reboot
    menu_item = gtk_image_menu_item_new_with_label("Reboot");
    set_item_icon(menu_item, "view-refresh");
    g_signal_connect(GTK_WIDGET(menu_item), "activate", G_CALLBACK(menuitem_click_handler),
        app.str_reboot_cmd);
    gtk_menu_shell_append(GTK_MENU_SHELL(app.main_menu), menu_item);

    // shutdown
    menu_item = gtk_image_menu_item_new_with_label("Shutdown");
    set_item_icon(menu_item, "system-shutdown");
    g_signal_connect(GTK_WIDGET(menu_item), "activate", G_CALLBACK(menuitem_click_handler),
        app.str_poweroff_cmd);
    gtk_menu_shell_append(GTK_MENU_SHELL(app.main_menu), menu_item);
}


/* display the menu */
gboolean popup_menu ()
{
    // set up callbacks
    g_signal_connect(GTK_WIDGET(app.main_menu), "size-allocate", G_CALLBACK(get_menu_size), NULL);
    g_signal_connect(GTK_MENU_SHELL(app.main_menu), "deactivate", G_CALLBACK(gtk_main_quit), NULL);
    g_signal_connect(GTK_MENU_SHELL(app.main_menu), "cancel", G_CALLBACK(gtk_main_quit), NULL);

    // set menu to visible
    gtk_widget_show_all(GTK_WIDGET(app.main_menu));

    // show menu
    if (app.show_at_mouse_pos == TRUE)
        gtk_menu_popup(GTK_MENU(app.main_menu), NULL, NULL, NULL, NULL, 0, 0);
    else
    {
        gtk_menu_popup(GTK_MENU(app.main_menu), NULL, NULL, menu_positioner, NULL, 0, 0);
        gtk_menu_reposition(GTK_MENU(app.main_menu));
    }

    return FALSE;
}


/* kill app after a while */
gboolean times_up ()
{
    exit(0);
    return FALSE;
}


/* delayed start */
gboolean delayed_run ()
{
    process_desktop_files();
    sort_list();
    build_menu();
    popup_menu();

    return FALSE;
}


/* Main program */
int main (int argc, char* argv[])
{
    gtk_init(&argc, &argv);

    // app timeout
    g_timeout_add(120000, times_up, NULL);

    init_app();
    process_config_file();

    if (app.delay_time < 100)
        app.delay_time = 100;

    // delay opening of menu
    g_timeout_add(app.delay_time, delayed_run, NULL);

    gtk_main();

    return 0;
}
